<?php
require_once('../includes/Developers.php');
header('Content-Type: application/json; charset=utf-8');

class InsuranceRenewalProcessor {
    private $db;
    private $response = [];
    private $params = [];
    private $couverture;
    private $couvertureOld;

    public function __construct() {
        $this->db = new Developers();
        $this->validateRequest();
    }

    private function validateRequest() {
        if ($_SERVER['REQUEST_METHOD'] != 'POST') {
            $this->sendErrorResponse('Invalid request method');
        }

        $requiredFields = [
            'uid_Id', 'uid_insId', 'uid_police', 'uid_proprio',
            'uid_effet', 'uid_echeance', 'uid_oldCouverture',
            'uid_couverture', 'uid_paymode', 'uid_agence'
        ];

        foreach ($requiredFields as $field) {
            if (!isset($_POST[$field])) {
                $this->sendErrorResponse("Missing required field: $field");
            }
        }

        $this->initializeParams();
    }

    private function sendErrorResponse(string $message): void {
        $this->response = ['error' => true, 'message' => $message];
        $this->sendResponse();
    }

    private function initializeParams() {
        $this->params = [
            'userId' => $_POST['uid_Id'],
            'insuranceId' => $_POST['uid_insId'],
            'police' => $_POST['uid_police'],
            'proprietaire' => $_POST['uid_proprio'],
            'Effet' => $_POST['uid_effet'],
            'Echeance' => $_POST['uid_echeance'],
            'couvertureOld' => (int)$_POST['uid_oldCouverture'],
            'couverture' => (int)$_POST['uid_couverture'],
            'paymode' => $_POST['uid_paymode'],
            'agencyV1' => $_POST['uid_agence']
        ];

        $this->couverture = $this->params['couverture'];
        $this->couvertureOld = $this->params['couvertureOld'];
    }

    public function processRenewal() {
        try {
            $this->validateCertificate();
            $this->validateNotePrime();
            $this->validatePoliceDetails();
            $this->validateMaxPolId();
            $this->validateAgencyDetails();
            $this->prepareCommonParams();
            $this->processGuarantees();
        } catch (Exception $e) {
            $this->sendErrorResponse($e->getMessage());
        }
    }

    private function validateCertificate() {
        $certificateDetails = $this->db->getAgCertificateValues("PBMO", $this->params['agencyV1'], "Ready");
        if (!$certificateDetails) {
            throw new Exception("Numero de certificat introuvable ou votre stock est epuisé");
        }
        $this->params['addCertificate'] = $certificateDetails['certificate_Id'];
    }

    private function validateNotePrime() {
        $noteprimeDetails = $this->db->getAgNotePrimeValues("PBMO", "Ready", $this->params['agencyV1']);
        if (!$noteprimeDetails) {
            throw new Exception("Numero de prime introuvable ou votre stock est epuisé");
        }
        $this->params['addNotePrime'] = $noteprimeDetails['notprime_Id'];
    }

    private function validatePoliceDetails() {
        $policeDetails = $this->db->getPoliceValues($this->params['insuranceId']);
        if (!$policeDetails) {
            throw new Exception("Référence de facture introuvable.");
        }

        $this->params = array_merge($this->params, [
            'numeroId' => $policeDetails['Numcontrat'],
            'idContrat' => $policeDetails['id_contrat'],
            'nemeroPol' => $policeDetails['Numpol'],
            'nemeroInv' => $policeDetails['facture'],
            'catAutomobile' => $policeDetails['CategoryAutomobile'],// Added this line  'catAutomobile' => $policeDetails['CategoryAutomobile'],'addCategoryAutomobile' => $policeDetails['CategoryAutomobile'],
            'plateId' => $policeDetails['Plaque'],
            'catPlatId' => $policeDetails['PlaqueCategor'],
            'model' => $policeDetails['ClassMarkModel'],
            'chassis' => $policeDetails['NumChasis'],
            'fabricat' => $policeDetails['FabricaYear'],
            'puissance' => $policeDetails['PuissanceCV'],
            'cabine' => $policeDetails['PaleSeatCabine'],
            'plato' => $policeDetails['SeatPlacePlato'],
            'usage' => $policeDetails['UsageAuto'],
            'preneurNom' => $policeDetails['preneur'],
            'proprio' => $policeDetails['Proprietaire'],
            'proprio_id' => $policeDetails['proprio_id'],
            'preneurNIF' => $policeDetails['NIF'],
            'preneurPhone' => $policeDetails['phone'],
            'preneurResidence' => $policeDetails['residence'],
            'preneurProvince' => $policeDetails['province'],
            'preneurPays' => $policeDetails['country'],
            'categoryProprio' => $policeDetails['CatProprietaire'],
            'carteRose' => $policeDetails['DateDelivrCarteRose'],
            'garanti' => $policeDetails['Garantie'],
            'valeurAss' => $policeDetails['valeur_assure'],
            'quotationAss' => $policeDetails['quotation_rate'],
            'userFullname' => $policeDetails['Creator'],
            'user_code' => $policeDetails['user_code'],
            'AgenceId' => $policeDetails['agence_id'],
            'agence' => $policeDetails['agence'],
            'AccountAgence' => $policeDetails['agence_account'],
            'username' => $policeDetails['username'],
            'ownerId' => $policeDetails['owner_id'],
            'Assurance_Id' => $policeDetails['Id']
        ]);
    }
 // 'addPoliceId' => $this->params['insuranceId']
    private function validateMaxPolId() {
        $maxDetails = $this->db->getMaxPolIdValues($this->params['police']);
        if (!$maxDetails) {
            throw new Exception("Référence de police introuvable.");
        }

        $this->params['contratId'] = $maxDetails['contract_Id'];
        $this->params['addPoliceId'] = $this->params['contratId'] + 1;
        $this->params['newContractPol'] = $this->params['nemeroPol'] . '.' . $this->params['addPoliceId'];
        $this->params['lock'] = 4;
        $this->params['renId'] = 2;
    }

    private function validateAgencyDetails() {
        $agencyDetails = $this->db->getClAgencyValues($this->params['userId']);
        if (!$agencyDetails) {
            throw new Exception("Votre Compte introuvable");
        }

        $this->params = array_merge($this->params, [
            'var_username' => $agencyDetails['username'],
            'var_userFullname' => $agencyDetails['fullname'],
            'var_agenceName' => $this->params['agencyV1'],
            'var_userCode' => $agencyDetails['user_id']
        ]);
    }

    private function prepareCommonParams() {
        $opReference = $this->db->getPinCodeNumber(20, false, false, true);
        $timestamp = time();

        $this->params = array_merge($this->params, [
            'currentDate' => date("Y-m-d"),
            'currentTime' => date("H:i:s"),
            'date_operation' => date("Y-m-d"),
            'emission' => date("Y-m-d"),
            'date_facture' => date("Y-m-d h:i:s"),
            'date_facture_signature' => date("Ymdhis"),
            'Year2' => date("y"),
            'op_reference_v1' => $opReference,
            'urlDetailsQrCode' => $opReference . $timestamp . '.' . date("Ymd") . '.' . date("His"),
             'Motif' => 'Renouvellement -'. $this->params['police'],
            'addContratId' => $this->params['newContractPol'],
            'addPolice' => $this->params['police'],
            'addDatteEffet' => $this->params['Effet'],
            'addDateEcheance' => $this->params['Echeance'],
            'addPeriode' => $this->params['couverture'],
            'addPlaque' => $this->params['plateId'],
            'addCategory' => $this->params['catPlatId'],
            'addClasseMarkModel' => $this->params['model'],
            'addChassis' => $this->params['chassis'],
            'addFabrication' => $this->params['fabricat'],
            'addPuissance' => $this->params['puissance'],
            'addPassagerCabine' => $this->params['cabine'],
            'addPassagerPlateau' => $this->params['plato'],
            'addUsage' => $this->params['usage'],
            'addProprio' => $this->params['preneurNom'],
            'addPreneur' => $this->params['proprio'],
            'addPreneurId' => $this->params['proprio_id'],
            'addPrenNIF' => $this->params['preneurNIF'],
            'addPrenPhone' => $this->params['preneurPhone'],
            'addPrenResidence' => $this->params['preneurResidence'],
            'addPrenProvince' => $this->params['preneurProvince'],
            'addPrenPays' => $this->params['preneurPays'],
            'addcategoryPren' => $this->params['categoryProprio'],
            'addCarteRose' => $this->params['carteRose'],
            'addValeurAjouter' => $this->params['valeurAss'],
            'addTaux' => $this->params['quotationAss'],
            'paymentMethod' => $this->params['paymode'],
            'SignatureDate' => date("Y-m-d h:i:s"),
            'ObrDateSignature' => date("Ymdhis")
        ]);
    }

    private function processGuarantees() {
        $garanty = trim($this->params['garanti']);
        if (empty($garanty)) {
            throw new Exception('Aucune garantie fournie');
        }

        $items = explode('+', $garanty);
        $processed = [];
        $feedbacks = [];
        $rcId = null;

        if (!$this->db->beginTransaction()) {
            throw new Exception("Could not start transaction");
        }

        try {
            foreach ($items as $item) {
                $item = trim($item);
                if (empty($item)) continue;

                $result = $this->processGuaranteeItem($item, $rcId);
                $rcId = $result['rcId'];
                $processed[] = $item;
                $feedbacks[] = "$item => {$result['feedback']} (Chaîne: " . implode('+', $processed) . ")";
            }

            if (!$this->db->commitTransaction()) {
                throw new Exception("Transaction commit failed");
            }

            $this->response = [
                'error' => false,
                'message' => 'Toutes les garanties ont été traitées avec succès.',
                'details' => $feedbacks,
                'processed_up_to' => implode('+', $processed)
            ];
        } catch (Exception $e) {
            $this->db->rollbackTransaction();
            $this->response = [
                'error' => true,
                'message' => "Erreur critique: " . $e->getMessage(),
                'details' => $feedbacks
            ];
        }

        $this->sendResponse();
    }

    private function processGuaranteeItem(string $item, ?int $currentRcId): array {
        $this->params['linked_rc_id'] = $currentRcId;
        $this->params['garantie'] = $item;
        
        // Build the new guarantee string
        $processedItems = $currentRcId ? explode('+', $this->params['garanti']) : [$item];
        $this->params['new_garantie'] = implode('+', array_unique($processedItems));
        
        // Calculate prime net values
        $primeNet = $this->getPrimeNetOrFail($item);
        $annualPrimeNet = $this->calculateAnnualPrimeNet($this->couvertureOld, $primeNet);
        $this->params['addPN'] = $this->calculateNewPrimeNet($this->couverture, $annualPrimeNet);

        // Map guarantee types to methods
        $methodMap = [
            'RC' => 'getAutomobileAgRC',
            'IOV' => 'getAutomobileAgIOV',
            'DM' => 'getAutomobileAgTDM',
            'INC' => 'getAutomobileAgINC',
            'VOL' => 'getAutomobileAgVOL'
        ];

        if (!isset($methodMap[$item])) {
            throw new Exception("Garantie inconnue: $item");
        }

        $methodName = $methodMap[$item];
        
        // Verify method exists
        if (!method_exists($this->db, $methodName)) {
            throw new Exception("Database method $methodName not implemented");
        }

        // Get parameters
        $params = $this->getMethodParams($methodName);
        
        // Debug logging
        error_log("Calling method: $methodName");
        error_log("Parameters count: " . count($params));
        error_log("Parameters: " . json_encode($params));
        
        // Call the method
        $result = call_user_func_array([$this->db, $methodName], $params);
        
        if ($result != 1 && $result != 4) {
            throw new Exception("Echec du traitement $item (code: $result)");
        }

        // Get the new RC ID
        $newRcId = $this->db->getMaxIdByPolice($this->params['addPolice']);
        if (!$newRcId) {
            throw new Exception("Impossible de récupérer l'ID du contrat renouvelé");
        }

        // Prepare feedback
        $feedback = ($result == 1) ? "$item renouvelé (ID: $newRcId)" : "$item déjà renouvelé (ID: $newRcId)";

        // Finalize process
        $finalize = $this->finalizeGuaranteeProcess();
        if (!$finalize['success']) {
            throw new Exception($finalize['message']);
        }

        return [
            'rcId' => $newRcId,
            'feedback' => $feedback . " | " . $finalize['message']
        ];
    }

    private function getPrimeNetOrFail(string $garantie): float {
        $pnDetails = $this->db->getPrimeNetByGarentyValues($this->params['nemeroInv'], $garantie);
        if (!$pnDetails) {
            throw new Exception("Référence de PN pour $garantie introuvable");
        }
        return (float)$pnDetails['primenet_Id'];
    }

    private function getMethodParams(string $method): array {
        // Base parameters for all methods
        $baseParams = [
            $this->params['addContratId'],       // 1. contratId
            $this->params['addPoliceId'],        // 2. policeId
            $this->params['addPolice'],          // 3. police
            $this->params['addDatteEffet'],      // 4. dateEffet
            $this->params['addDateEcheance'],    // 5. dateEcheance
            $this->params['addPeriode'],         // 6. periode
            $this->params['addCertificate'],     // 7. certificate
            $this->params['addNotePrime'],       // 8. notePrime
            $this->params['catAutomobile'], // 9. categoryAutomobile
            $this->params['addPlaque'],          // 10. plaque
            $this->params['addCategory'],        // 11. category
            $this->params['addClasseMarkModel'], // 12. classeMarkModel
            $this->params['addChassis'],         // 13. chassis
            $this->params['addFabrication'],     // 14. fabrication
            $this->params['addPuissance'],       // 15. puissance
            $this->params['addPassagerCabine'],  // 16. passagerCabine
            $this->params['addPassagerPlateau'], // 17. passagerPlateau
            $this->params['addUsage'],           // 18. usage
            $this->params['addProprio'],         // 19. proprio
            $this->params['addPreneur'],         // 20. preneur
            $this->params['addPreneurId'],       // 21. preneurId
            $this->params['addPrenNIF'],         // 22. prenNIF
            $this->params['addPrenPhone'],       // 23. prenPhone
            $this->params['addPrenResidence'],   // 24. prenResidence
            $this->params['addPrenProvince'],    // 25. prenProvince
            $this->params['addPrenPays'],        // 26. prenPays
            $this->params['addcategoryPren'],    // 27. categoryPren
            $this->params['addCarteRose'],       // 28. carteRose
            $this->params['addValeurAjouter'],   // 29. valeurAjouter
            $this->params['addTaux'],            // 30. taux
            $this->params['userFullname'],       // 31. userFullname
            $this->params['user_code'],          // 32. userCode
            $this->params['AgenceId'],           // 33. agenceId
            $this->params['agence'],             // 34. agence
            $this->params['AccountAgence'],      // 35. accountAgence
            $this->params['username'],           // 36. username
            $this->params['ownerId'],            // 37. ownerId
            $this->params['garantie'],           // 38. garantie
            $this->params['addPN'],              // 39. pnId
            $this->params['paymentMethod'],      // 40. paymentMethod
            $this->params['Year2'],              // 41. year2
            $this->params['SignatureDate'],      // 42. signatureDate
            $this->params['ObrDateSignature'],   // 43. obrDateSignature
            $this->params['urlDetailsQrCode']    // 44. qrCodeUrl
        ];

        // Additional parameters for specific methods (not RC)
        if ($method !== 'getAutomobileAgRC') {
            $baseParams[] = $this->params['new_garantie'];   // 45. newGarantie
            $baseParams[] = $this->params['linked_rc_id'];   // 46. linked_rc_id
        }

        return $baseParams;
    }

    private function calculateAnnualPrimeNet(int $couvertureOld, float $primeNet): float {
        switch ($couvertureOld) {
            case 3: return ($primeNet * 4 * 100) / 103;
            case 6: return ($primeNet * 2 * 100) / 102;
            case 12: return $primeNet;
            default: return 0;
        }
    }

    private function calculateNewPrimeNet(int $couverture, float $annualPrimeNet): float {
        switch ($couverture) {
            case 3: return ($annualPrimeNet * 1.03) / 4;
            case 6: return ($annualPrimeNet * 1.02) / 2;
            case 12: return $annualPrimeNet;
            default: return 0;
        }
    }

    private function finalizeGuaranteeProcess(): array {
        if ($this->db->getReport(
            $this->params['Motif'], 
            $this->params['var_agenceName'], 
            $this->params['var_username'], 
            $this->params['date_operation']
        ) != 1) {
            return ['success' => false, 'message' => "Erreur lors du rapport"];
        }

        // if ($this->db->getUpdtSelfBuy(
        //     $this->params['addCertificate'], 
        //     $this->params['userId'], 
        //     $this->params['user_code'], 
        //     $this->params['addPoliceId'], 
        //     $this->params['addPreneur']
        // ) != 1) {
        //     return ['success' => false, 'message' => "Erreur lors de la mise à jour"];
        // }

        if ($this->db->getAgAssurUpt(
            $this->params['Assurance_Id'], 
            $this->params['lock'], 
            $this->params['renId'], 
            $this->params['var_agenceName']
        ) != 1) {
            return ['success' => false, 'message' => "Erreur lors de la mise à jour de l'assurance"];
        }

        return [
            'success' => true,
            'message' => "Le contrat est renouvelé, veuillez continuer pour payer l'assurance"
        ];
    }

    private function sendResponse(): void {
        echo json_encode($this->response);
        exit;
    }
}

$processor = new InsuranceRenewalProcessor();
$processor->processRenewal();