<?php
// ============================================================================
// LOGIN HANDLER - OPTIMIZED VERSION
// ============================================================================

// Include configuration (handles environment, database, security)
require_once('settings/config.php');

// Start session (configured in config.php)
session_start();

// ============================================================================
// CSRF PROTECTION
// ============================================================================

// Generate CSRF token if not exists
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// CSRF Token Validation
function validateCSRFToken() {
    if (empty($_POST['csrf_token']) || empty($_SESSION['csrf_token'])) {
        return false;
    }
    
    // Use hash_equals for timing attack protection
    return hash_equals($_SESSION['csrf_token'], $_POST['csrf_token']);
}

// ============================================================================
// RATE LIMITING
// ============================================================================

/**
 * Check rate limiting for login attempts
 */
function checkLoginRateLimit($username, $conn) {
    $ip = $_SERVER['REMOTE_ADDR'];
    $maxAttempts = 5;
    $lockTime = 900; // 15 minutes
    $windowTime = 300; // 5 minutes window
    
    try {
        // Clean old attempts
        $cleanTime = time() - $windowTime;
        mysqli_query($conn, "DELETE FROM login_attempts WHERE timestamp < $cleanTime");
        
        // Count recent attempts by IP
        $ipQuery = "SELECT COUNT(*) as attempts FROM login_attempts 
                    WHERE ip_address = ? AND timestamp > ?";
        $stmt = mysqli_prepare($conn, $ipQuery);
        $timeThreshold = time() - $windowTime;
        mysqli_stmt_bind_param($stmt, "si", $ip, $timeThreshold);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        $ipAttempts = mysqli_fetch_assoc($result)['attempts'] ?? 0;
        mysqli_stmt_close($stmt);
        
        if ($ipAttempts >= $maxAttempts) {
            return [
                'allowed' => false,
                'message' => 'Trop de tentatives. Veuillez réessayer dans 5 minutes.',
                'wait_time' => $windowTime
            ];
        }
        
        // Check if username exists and get failed attempts
        $userQuery = "SELECT failed_attempts, locked_until FROM code_users 
                      WHERE username = ? AND status = 'active'";
        $stmt = mysqli_prepare($conn, $userQuery);
        mysqli_stmt_bind_param($stmt, "s", $username);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        if ($user = mysqli_fetch_assoc($result)) {
            // Check if account is locked
            if ($user['locked_until'] && strtotime($user['locked_until']) > time()) {
                $remaining = strtotime($user['locked_until']) - time();
                return [
                    'allowed' => false,
                    'message' => 'Compte temporairement bloqué. Réessayez dans ' . ceil($remaining/60) . ' minutes.',
                    'wait_time' => $remaining
                ];
            }
        }
        
        mysqli_stmt_close($stmt);
        return ['allowed' => true];
        
    } catch (Exception $e) {
        error_log("Rate limit check error: " . $e->getMessage());
        return ['allowed' => true]; // Allow on error to not block legitimate users
    }
}

// ============================================================================
// INPUT VALIDATION
// ============================================================================

/**
 * Validate login input
 */
function validateLoginInput($username, $password) {
    // Trim inputs
    $username = trim($username);
    $password = trim($password);
    
    // Check for empty values
    if (empty($username) || empty($password)) {
        return ['valid' => false, 'error' => 'Nom d\'utilisateur et mot de passe requis'];
    }
    
    // Validate username format (alphanumeric and underscore)
    if (!preg_match('/^[A-Za-z0-9_]{3,50}$/', $username)) {
        return ['valid' => false, 'error' => 'Format de nom d\'utilisateur invalide'];
    }
    
    // Check password length
    if (strlen($password) < 8 || strlen($password) > 72) {
        return ['valid' => false, 'error' => 'Le mot de passe doit contenir entre 8 et 72 caractères'];
    }
    
    return ['valid' => true, 'username' => $username, 'password' => $password];
}

// ============================================================================
// PASSWORD VERIFICATION
// ============================================================================

/**
 * Verify password and auto-upgrade from MD5 to bcrypt
 */
function verifyPassword($inputPassword, $storedHash, $conn, $userId = null) {
    // If using old MD5 passwords (you should migrate from this!)
    if (strlen($storedHash) === 32 && ctype_xdigit($storedHash)) {
        // MD5 detected - verify but flag for update
        if (md5($inputPassword) === $storedHash) {
            // Auto-migrate to bcrypt on successful login
            $newHash = password_hash($inputPassword, PASSWORD_DEFAULT);
            $updateQuery = "UPDATE code_users SET password = ?, password_upgraded = 1 WHERE Id = ?";
            $stmt = mysqli_prepare($conn, $updateQuery);
            mysqli_stmt_bind_param($stmt, "si", $newHash, $userId);
            mysqli_stmt_execute($stmt);
            mysqli_stmt_close($stmt);
            return true;
        }
        return false;
    }
    
    // Modern password_hash verification
    return password_verify($inputPassword, $storedHash);
}

// ============================================================================
// LOGIN ATTEMPT LOGGING
// ============================================================================

/**
 * Log login attempts for security monitoring
 */
function logLoginAttempt($conn, $username, $ip, $success, $userId = null) {
    try {
        $timestamp = time();
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
        
        // Log attempt in database
        $query = "INSERT INTO login_attempts (username, ip_address, user_agent, success, timestamp, user_id) 
                  VALUES (?, ?, ?, ?, ?, ?)";
        $stmt = mysqli_prepare($conn, $query);
        $successInt = $success ? 1 : 0;
        mysqli_stmt_bind_param($stmt, "sssisi", $username, $ip, $userAgent, $successInt, $timestamp, $userId);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
        
        // Update user's failed attempts counter on failed login
        if (!$success && $userId) {
            $updateQuery = "UPDATE code_users SET failed_attempts = failed_attempts + 1, 
                            last_failed_attempt = NOW() WHERE Id = ?";
            $stmt = mysqli_prepare($conn, $updateQuery);
            mysqli_stmt_bind_param($stmt, "i", $userId);
            mysqli_stmt_execute($stmt);
            mysqli_stmt_close($stmt);
            
            // Lock account after 5 failed attempts
            $checkQuery = "SELECT failed_attempts FROM code_users WHERE Id = ?";
            $stmt = mysqli_prepare($conn, $checkQuery);
            mysqli_stmt_bind_param($stmt, "i", $userId);
            mysqli_stmt_execute($stmt);
            $result = mysqli_stmt_get_result($stmt);
            $user = mysqli_fetch_assoc($result);
            mysqli_stmt_close($stmt);
            
            if ($user && $user['failed_attempts'] >= 5) {
                $lockUntil = date('Y-m-d H:i:s', time() + 900); // 15 minutes lock
                $lockQuery = "UPDATE code_users SET locked_until = ? WHERE Id = ?";
                $stmt = mysqli_prepare($conn, $lockQuery);
                mysqli_stmt_bind_param($stmt, "si", $lockUntil, $userId);
                mysqli_stmt_execute($stmt);
                mysqli_stmt_close($stmt);
            }
        }
    } catch (Exception $e) {
        // Log error but don't break login process
        error_log("Login attempt logging error: " . $e->getMessage());
    }
}

// ============================================================================
// SESSION MANAGEMENT
// ============================================================================

/**
 * Set user session after successful login
 */
function setUserSession($userData) {
    // Regenerate session ID to prevent fixation
    session_regenerate_id(true);
    
    // Set session variables
    $_SESSION["persId"] = $userData['Id'];
    $_SESSION["user_id"] = $userData['user_id'];
    $_SESSION["fullname"] = htmlspecialchars($userData['fullname'], ENT_QUOTES, 'UTF-8');
    $_SESSION["username"] = htmlspecialchars($userData['username'], ENT_QUOTES, 'UTF-8');
    $_SESSION["category"] = htmlspecialchars($userData['category'], ENT_QUOTES, 'UTF-8');
    $_SESSION["agence"] = htmlspecialchars($userData['agence'], ENT_QUOTES, 'UTF-8');
    $_SESSION["region_details"] = htmlspecialchars($userData['region_details'] ?? '', ENT_QUOTES, 'UTF-8');
    $_SESSION["auth"] = true;
    $_SESSION["login_time"] = time();
    $_SESSION["session_expire"] = time() + (40 * 60); // 40 minutes
    $_SESSION["last_activity"] = time();
    
    // Update CSRF token after login
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    
    // Store user agent for session validation
    $_SESSION['user_agent'] = $_SERVER['HTTP_USER_AGENT'] ?? '';
    
    // Store IP for session validation (first 3 octets only for privacy)
    $ip = $_SERVER['REMOTE_ADDR'] ?? '';
    $_SESSION['login_ip'] = $ip;
}

// ============================================================================
// REDIRECTION MAP
// ============================================================================

/**
 * Get redirect URL based on user category
 */
function getRedirectUrl($category) {
    $redirectMap = [
        'Super' => 'head_dashboard',
        'User_Siege' => 'user_dashboard',
        'Assistance' => 'ass_cust_dashboard',
        'Accounter' => 'Account_dashboard',
        'Underwritter' => 'Souscr_dashboard',
        'Souscription' => 'souscription_dashboard',
        'Member' => 'member_dashboard',
        'Administrator' => 'admin_dashboard',
        'Developer' => 'main_dashboard',
        'OBR' => 'tax_verification',
        'Rootar' => 'xyz_dashboard',
        'Sinistre' => 'sinistre_dashboard',
        'Support' => 'support_dashboard',
        'Siege' => 'ag_sousc_dashboard',
        'Caisse' => 'ag_sousc_dashboard_cashier',
        'Manager' => 'sys_mng_dashboard',
        'Supervisor' => 'sys_spv_dashboard'
    ];
    
    return $redirectMap[$category] ?? './';
}

// ============================================================================
// MAIN LOGIN PROCESSING
// ============================================================================

// Only process POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['open_ssl_con'])) {
    
    $conn = null;
    $stmt = null;
    
    try {
        // 1. Validate CSRF token
        if (!validateCSRFToken()) {
            throw new Exception('Session invalide. Veuillez rafraîchir la page.');
        }
        
        // 2. Validate input
        $validation = validateLoginInput($_POST['username'] ?? '', $_POST['password'] ?? '');
        if (!$validation['valid']) {
            throw new Exception($validation['error']);
        }
        
        $username = $validation['username'];
        $password = $validation['password'];
        $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
        
        // 3. Create database connection
        $conn = createDatabaseConnection();
        
        // 4. Check rate limiting
        $rateLimit = checkLoginRateLimit($username, $conn);
        if (!$rateLimit['allowed']) {
            throw new Exception($rateLimit['message']);
        }
        
        // 5. Prepare and execute query with parameterized statement
        $query = "SELECT Id, user_id, username, password, fullname, category, agence, 
                  region_details, failed_attempts, locked_until, status, email
                  FROM code_users 
                  WHERE username = ? AND status = 'active'";
        
        $stmt = mysqli_prepare($conn, $query);
        
        if (!$stmt) {
            throw new Exception('Erreur système. Veuillez réessayer plus tard.');
        }
        
        mysqli_stmt_bind_param($stmt, "s", $username);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        // 6. Check if user exists
        if (mysqli_num_rows($result) === 1) {
            $user = mysqli_fetch_assoc($result);
            
            // Check if account is locked
            if ($user['locked_until'] && strtotime($user['locked_until']) > time()) {
                $remaining = strtotime($user['locked_until']) - time();
                throw new Exception('Compte bloqué. Réessayez dans ' . ceil($remaining/60) . ' minutes.');
            }
            
            // 7. Verify password
            if (verifyPassword($password, $user['password'], $conn, $user['Id'])) {
                // Successful login
                
                // Reset failed attempts
                $resetQuery = "UPDATE code_users SET 
                              failed_attempts = 0, 
                              locked_until = NULL, 
                              last_login = NOW(),
                              login_count = IFNULL(login_count, 0) + 1
                              WHERE Id = ?";
                $resetStmt = mysqli_prepare($conn, $resetQuery);
                mysqli_stmt_bind_param($resetStmt, "i", $user['Id']);
                mysqli_stmt_execute($resetStmt);
                mysqli_stmt_close($resetStmt);
                
                // Log successful attempt
                logLoginAttempt($conn, $username, $ip, true, $user['Id']);
                
                // Set user session
                setUserSession($user);
                
                // Get redirect URL
                $redirectUrl = getRedirectUrl($user['category']);
                
                // Close database connections
                mysqli_stmt_close($stmt);
                mysqli_close($conn);
                
                // Redirect to dashboard
                header("Location: $redirectUrl");
                exit();
                
            } else {
                // Failed password verification
                logLoginAttempt($conn, $username, $ip, false, $user['Id']);
            }
        }
        
        // 8. Generic error for failed login (don't reveal if user exists)
        logLoginAttempt($conn, $username, $ip, false);
        
        // Add small delay to slow down brute force (100-500ms)
        usleep(rand(100000, 500000));
        
        throw new Exception('Nom d\'utilisateur ou mot de passe incorrect');
        
    } catch (Exception $e) {
        // Clean up database connections
        if ($stmt) {
            mysqli_stmt_close($stmt);
        }
        if ($conn) {
            mysqli_close($conn);
        }
        
        // Set error message
        $_SESSION['error'] = $e->getMessage();
        
        // Store username for form repopulation (security: only on validation errors)
        if (strpos($e->getMessage(), 'Nom d\'utilisateur') === false) {
            $_SESSION['login_attempt_username'] = htmlspecialchars($username ?? '');
        }
        
        // Redirect back to login page
        header('Location: ./');
        exit();
    }
    
} else {
    // Not a POST request or invalid form submission
    header('Location: ./');
    exit();
}

// ============================================================================
// END OF FILE
// ============================================================================