<?php
// ============================================================================
// SECURITY ENHANCEMENTS - START
// ============================================================================

// Force HTTPS (remove comment if you have HTTPS)
if (empty($_SERVER['HTTPS']) || $_SERVER['HTTPS'] === 'off') {
    header('Location: https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']);
    exit();
}

// Security headers
header("X-Frame-Options: DENY");
header("X-Content-Type-Options: nosniff");
header("X-XSS-Protection: 1; mode=block");
header("Referrer-Policy: strict-origin-when-cross-origin");

// Content-Security-Policy - adjust based on your needs
// header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline' 'unsafe-eval'; style-src 'self' 'unsafe-inline';");

// Set secure session parameters before starting session
ini_set('session.cookie_httponly', 1);
ini_set('session.cookie_secure', 1); // Enable only with HTTPS
ini_set('session.cookie_samesite', 'Strict');
ini_set('session.use_strict_mode', 1);
ini_set('session.use_only_cookies', 1);

// Start session
session_start();

// Regenerate session ID periodically for security
if (!isset($_SESSION['created'])) {
    $_SESSION['created'] = time();
} elseif (time() - $_SESSION['created'] > 1800) { // 30 minutes
    session_regenerate_id(true);
    $_SESSION['created'] = time();
}

// ============================================================================
// CONFIGURATION & CSRF PROTECTION
// ============================================================================

require_once('settings/config.php');

// CSRF Token Validation
function validateCSRFToken() {
    if (empty($_POST['csrf_token']) || empty($_SESSION['csrf_token'])) {
        return false;
    }
    
    // Use hash_equals for timing attack protection
    if (!hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        return false;
    }
    
    return true;
}

// Generate CSRF token if not exists
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// ============================================================================
// RATE LIMITING FUNCTION
// ============================================================================

function checkRateLimit($username, $conn) {
    $ip = $_SERVER['REMOTE_ADDR'];
    $maxAttempts = 5;
    $lockTime = 900; // 15 minutes in seconds
    $windowTime = 300; // 5 minutes window
    
    // Clean old attempts
    $cleanTime = time() - $windowTime;
    mysqli_query($conn, "DELETE FROM login_attempts WHERE timestamp < $cleanTime");
    
    // Count recent attempts by IP
    $ipQuery = "SELECT COUNT(*) as attempts FROM login_attempts 
                WHERE ip_address = ? AND timestamp > ?";
    $stmt = mysqli_prepare($conn, $ipQuery);
    $timeThreshold = time() - $windowTime;
    mysqli_stmt_bind_param($stmt, "si", $ip, $timeThreshold);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $ipAttempts = mysqli_fetch_assoc($result)['attempts'];
    mysqli_stmt_close($stmt);
    
    if ($ipAttempts >= $maxAttempts) {
        return [
            'allowed' => false,
            'message' => 'Trop de tentatives. Veuillez réessayer dans 5 minutes.',
            'wait_time' => $windowTime
        ];
    }
    
    // Check if username exists and get failed attempts
    $userQuery = "SELECT failed_attempts, locked_until FROM code_users 
                  WHERE username = ?";
    $stmt = mysqli_prepare($conn, $userQuery);
    mysqli_stmt_bind_param($stmt, "s", $username);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if ($user = mysqli_fetch_assoc($result)) {
        // Check if account is locked
        if ($user['locked_until'] && strtotime($user['locked_until']) > time()) {
            $remaining = strtotime($user['locked_until']) - time();
            return [
                'allowed' => false,
                'message' => 'Compte temporairement bloqué. Réessayez dans ' . ceil($remaining/60) . ' minutes.',
                'wait_time' => $remaining
            ];
        }
    }
    
    mysqli_stmt_close($stmt);
    return ['allowed' => true];
}

// ============================================================================
// INPUT VALIDATION & SANITIZATION
// ============================================================================

function validateLoginInput($username, $password) {
    // Trim inputs
    $username = trim($username);
    $password = trim($password);
    
    // Check for empty values
    if (empty($username) || empty($password)) {
        return ['valid' => false, 'error' => 'Nom d\'utilisateur et mot de passe requis'];
    }
    
    // Validate username format (alphanumeric and underscore)
    if (!preg_match('/^[A-Za-z0-9_]{3,50}$/', $username)) {
        return ['valid' => false, 'error' => 'Format de nom d\'utilisateur invalide'];
    }
    
    // Check password length
    if (strlen($password) < 8 || strlen($password) > 72) {
        return ['valid' => false, 'error' => 'Le mot de passe doit contenir entre 8 et 72 caractères'];
    }
    
    return ['valid' => true, 'username' => $username, 'password' => $password];
}

// ============================================================================
// PASSWORD VERIFICATION
// ============================================================================

function verifyPassword($inputPassword, $storedHash, $conn, $userId = null) {
    // If using old MD5 passwords (you should migrate from this!)
    if (strlen($storedHash) === 32 && ctype_xdigit($storedHash)) {
        // MD5 detected - verify but flag for update
        if (md5($inputPassword) === $storedHash) {
            // Auto-migrate to bcrypt on successful login
            $newHash = password_hash($inputPassword, PASSWORD_DEFAULT);
            $updateQuery = "UPDATE code_users SET password = ?, password_upgraded = 1 WHERE user_id = ?";
            $stmt = mysqli_prepare($conn, $updateQuery);
            mysqli_stmt_bind_param($stmt, "si", $newHash, $userId);
            mysqli_stmt_execute($stmt);
            mysqli_stmt_close($stmt);
            return true;
        }
        return false;
    }
    
    // Modern password_hash verification
    return password_verify($inputPassword, $storedHash);
}

// ============================================================================
// LOGIN ATTEMPT LOGGING
// ============================================================================

function logLoginAttempt($conn, $username, $ip, $success, $userId = null) {
    $timestamp = time();
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    
    // Log attempt in database
    $query = "INSERT INTO login_attempts (username, ip_address, user_agent, success, timestamp, user_id) 
              VALUES (?, ?, ?, ?, ?, ?)";
    $stmt = mysqli_prepare($conn, $query);
    $successInt = $success ? 1 : 0;
    mysqli_stmt_bind_param($stmt, "sssisi", $username, $ip, $userAgent, $successInt, $timestamp, $userId);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
    
    // Also update user's failed attempts counter
    if (!$success && $userId) {
        $updateQuery = "UPDATE code_users SET failed_attempts = failed_attempts + 1, 
                        last_failed_attempt = NOW() WHERE user_id = ?";
        $stmt = mysqli_prepare($conn, $updateQuery);
        mysqli_stmt_bind_param($stmt, "i", $userId);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
        
        // Lock account after 5 failed attempts
        $checkQuery = "SELECT failed_attempts FROM code_users WHERE user_id = ?";
        $stmt = mysqli_prepare($conn, $checkQuery);
        mysqli_stmt_bind_param($stmt, "i", $userId);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        $user = mysqli_fetch_assoc($result);
        mysqli_stmt_close($stmt);
        
        if ($user['failed_attempts'] >= 5) {
            $lockUntil = date('Y-m-d H:i:s', time() + 900); // 15 minutes lock
            $lockQuery = "UPDATE code_users SET locked_until = ? WHERE user_id = ?";
            $stmt = mysqli_prepare($conn, $lockQuery);
            mysqli_stmt_bind_param($stmt, "si", $lockUntil, $userId);
            mysqli_stmt_execute($stmt);
            mysqli_stmt_close($stmt);
        }
    }
}

// ============================================================================
// MAIN LOGIN PROCESSING
// ============================================================================

// Only process POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['open_ssl_con'])) {
    
    // 1. Validate CSRF token
    if (!validateCSRFToken()) {
        $_SESSION['error'] = 'Session invalide. Veuillez rafraîchir la page.';
        header('Location: ./');
        exit();
    }
    
    // 2. Validate input
    $validation = validateLoginInput($_POST['username'] ?? '', $_POST['password'] ?? '');
    if (!$validation['valid']) {
        $_SESSION['error'] = $validation['error'];
        header('Location: ./');
        exit();
    }
    
    $username = $validation['username'];
    $password = $validation['password'];
    $ip = $_SERVER['REMOTE_ADDR'];
    
    // 3. Check rate limiting
    $rateLimit = checkRateLimit($username, $conn);
    if (!$rateLimit['allowed']) {
        $_SESSION['error'] = $rateLimit['message'];
        header('Location: ./');
        exit();
    }
    
    // 4. Prepare and execute query with parameterized statement
    $query = "SELECT user_id, username, password, fullname, category, agence, 
              region_details, failed_attempts, locked_until, status 
              FROM code_users 
              WHERE username = ? AND status = 'active'";
    
    $stmt = mysqli_prepare($conn, $query);
    
    if (!$stmt) {
        error_log("Login prepare failed: " . mysqli_error($conn));
        $_SESSION['error'] = 'Erreur système. Veuillez réessayer plus tard.';
        header('Location: ./');
        exit();
    }
    
    mysqli_stmt_bind_param($stmt, "s", $username);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    // 5. Check if user exists
    if (mysqli_num_rows($result) === 1) {
        $row = mysqli_fetch_assoc($result);
        
        // Check if account is locked
        if ($row['locked_until'] && strtotime($row['locked_until']) > time()) {
            $remaining = strtotime($row['locked_until']) - time();
            $_SESSION['error'] = 'Compte bloqué. Réessayez dans ' . ceil($remaining/60) . ' minutes.';
            logLoginAttempt($conn, $username, $ip, false, $row['user_id']);
            header('Location: ./');
            exit();
        }
        
        // 6. Verify password
        if (verifyPassword($password, $row['password'], $conn, $row['user_id'])) {
            // Successful login
            
            // Reset failed attempts
            $resetQuery = "UPDATE code_users SET failed_attempts = 0, locked_until = NULL, 
                          last_login = NOW() WHERE user_id = ?";
            $resetStmt = mysqli_prepare($conn, $resetQuery);
            mysqli_stmt_bind_param($resetStmt, "i", $row['user_id']);
            mysqli_stmt_execute($resetStmt);
            mysqli_stmt_close($resetStmt);
            
            // Log successful attempt
            logLoginAttempt($conn, $username, $ip, true, $row['user_id']);
            
            // Regenerate session ID to prevent fixation
            session_regenerate_id(true);
            
            // Set session variables
            $_SESSION["user_id"] = $row['user_id'];
            $_SESSION["fullname"] = htmlspecialchars($row['fullname'], ENT_QUOTES, 'UTF-8');
            $_SESSION["username"] = htmlspecialchars($row['username'], ENT_QUOTES, 'UTF-8');
            $_SESSION["category"] = htmlspecialchars($row['category'], ENT_QUOTES, 'UTF-8');
            $_SESSION["agence"] = htmlspecialchars($row['agence'], ENT_QUOTES, 'UTF-8');
            $_SESSION["region_details"] = htmlspecialchars($row['region_details'] ?? '', ENT_QUOTES, 'UTF-8');
            $_SESSION["auth"] = true;
            $_SESSION["login_time"] = time();
            $_SESSION["session_expire"] = time() + (40 * 60); // 40 minutes
            
            // Update CSRF token after login
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
            
            // 7. Redirect based on category
            $redirectMap = [
                'Super' => 'head_dashboard',
                'User_Siege' => 'user_dashboard',
                'Assistance' => 'ass_cust_dashboard',
                'Accounter' => 'Account_dashboard',
                'Underwritter' => 'Souscr_dashboard',
                'Souscription' => 'souscription_dashboard',
                'Member' => 'member_dashboard',
                'Administrator' => 'admin_dashboard',
                'Developer' => 'main_dashboard',
                'OBR' => 'tax_verification',
                'Rootar' => 'xyz_dashboard',
                'Sinistre' => 'sinistre_dashboard',
                'Support' => 'support_dashboard',
                'Siege' => 'ag_sousc_dashboard',
                'Caisse' => 'ag_sousc_dashboard_cashier',
                'Manager' => 'sys_mng_dashboard',
                'Supervisor' => 'sys_spv_dashboard'
            ];
            
            $redirect = $redirectMap[$row['category']] ?? './';
            header("Location: $redirect");
            exit();
            
        } else {
            // Failed password verification
            logLoginAttempt($conn, $username, $ip, false, $row['user_id']);
        }
    }
    
    // 8. Generic error for failed login (don't reveal if user exists)
    // Log failed attempt without user ID (user not found)
    logLoginAttempt($conn, $username, $ip, false);
    
    // Small delay to slow down brute force (100-500ms)
    usleep(rand(100000, 500000));
    
    $_SESSION['error'] = 'Nom d\'utilisateur ou mot de passe incorrect';
    header('Location: ./');
    exit();
    
} else {
    // Not a POST request or invalid form submission
    header('Location: ./');
    exit();
}

// Close statement if still open
if (isset($stmt) && $stmt) {
    mysqli_stmt_close($stmt);
}
?>