<?php
// ============================================================================
// FORGOT PASSWORD SYSTEM
// ============================================================================

// 1. CONFIGURE SESSION SETTINGS FIRST
ini_set('session.cookie_httponly', 1);
ini_set('session.cookie_secure', 1); // Enable only with HTTPS
ini_set('session.cookie_samesite', 'Strict');
ini_set('session.use_strict_mode', 1);
ini_set('session.use_only_cookies', 1);

// 2. SET SECURITY HEADERS (before any output)
header("X-Frame-Options: DENY");
header("X-Content-Type-Options: nosniff");
header("X-XSS-Protection: 1; mode=block");
header("Referrer-Policy: strict-origin-when-cross-origin");

// 3. START SESSION
session_start();

// 4. INCLUDE CONFIG
require_once('settings/config.php');

// 5. GENERATE CSRF TOKEN
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// ============================================================================
// CONFIGURATION
// ============================================================================
define('RESET_TOKEN_EXPIRY', 3600); // 1 hour
define('MAX_RESET_ATTEMPTS', 3);
define('RESET_COOLDOWN', 900); // 15 minutes
define('MIN_PASSWORD_LENGTH', 8);

// ============================================================================
// HELPER FUNCTIONS
// ============================================================================

/**
 * Validate email format
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

/**
 * Generate secure random token
 */
function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

/**
 * Check rate limiting for password reset
 */
function checkResetRateLimit($email, $conn) {
    $ip = $_SERVER['REMOTE_ADDR'];
    $now = time();
    
    // Clean old attempts
    $cleanTime = $now - RESET_COOLDOWN;
    mysqli_query($conn, "DELETE FROM password_reset_attempts WHERE timestamp < $cleanTime");
    
    // Check attempts by IP
    $ipQuery = "SELECT COUNT(*) as attempts FROM password_reset_attempts 
                WHERE ip_address = ? AND timestamp > ?";
    $stmt = mysqli_prepare($conn, $ipQuery);
    $timeThreshold = $now - RESET_COOLDOWN;
    mysqli_stmt_bind_param($stmt, "si", $ip, $timeThreshold);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $ipAttempts = mysqli_fetch_assoc($result)['attempts'];
    mysqli_stmt_close($stmt);
    
    if ($ipAttempts >= MAX_RESET_ATTEMPTS) {
        return [
            'allowed' => false,
            'message' => 'Trop de demandes. Veuillez réessayer dans 15 minutes.',
            'wait_time' => RESET_COOLDOWN
        ];
    }
    
    // Check attempts by email
    $emailQuery = "SELECT COUNT(*) as attempts FROM password_reset_attempts 
                   WHERE email = ? AND timestamp > ?";
    $stmt = mysqli_prepare($conn, $emailQuery);
    mysqli_stmt_bind_param($stmt, "si", $email, $timeThreshold);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $emailAttempts = mysqli_fetch_assoc($result)['attempts'];
    mysqli_stmt_close($stmt);
    
    if ($emailAttempts >= MAX_RESET_ATTEMPTS) {
        return [
            'allowed' => false,
            'message' => 'Trop de demandes pour cet email. Veuillez réessayer dans 15 minutes.',
            'wait_time' => RESET_COOLDOWN
        ];
    }
    
    return ['allowed' => true];
}

/**
 * Log reset attempt
 */
function logResetAttempt($conn, $email, $success) {
    $ip = $_SERVER['REMOTE_ADDR'];
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    $timestamp = time();
    
    $query = "INSERT INTO password_reset_attempts (email, ip_address, user_agent, success, timestamp) 
              VALUES (?, ?, ?, ?, ?)";
    $stmt = mysqli_prepare($conn, $query);
    $successInt = $success ? 1 : 0;
    mysqli_stmt_bind_param($stmt, "sssii", $email, $ip, $userAgent, $successInt, $timestamp);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
}

/**
 * Send password reset email
 */
function sendResetEmail($to, $name, $token) {
    $resetLink = "https://" . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . "/forgot_password.php?token=" . urlencode($token);
    $expiryTime = date('H:i', time() + RESET_TOKEN_EXPIRY);
    
    $subject = "MOBIC GLOBAL - Réinitialisation de votre mot de passe";
    
    $message = "
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset='UTF-8'>
        <title>Réinitialisation du mot de passe</title>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; text-align: center; border-radius: 10px 10px 0 0; }
            .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px; }
            .button { display: inline-block; background: #667eea; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; margin: 20px 0; }
            .warning { background: #fff3cd; border-left: 4px solid #ffc107; padding: 10px; margin: 20px 0; }
            .footer { text-align: center; margin-top: 30px; font-size: 12px; color: #666; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h2>MOBIC GLOBAL</h2>
            </div>
            <div class='content'>
                <p>Bonjour <strong>$name</strong>,</p>
                
                <p>Nous avons reçu une demande de réinitialisation de votre mot de passe MOBIC GLOBAL.</p>
                
                <p style='text-align: center;'>
                    <a href='$resetLink' class='button'>Réinitialiser mon mot de passe</a>
                </p>
                
                <p>Ou copiez-collez ce lien dans votre navigateur :</p>
                <p><code>$resetLink</code></p>
                
                <div class='warning'>
                    <strong>⚠️ Important :</strong>
                    <ul>
                        <li>Ce lien expirera à <strong>$expiryTime</strong> (dans 1 heure)</li>
                        <li>Si vous n'avez pas demandé cette réinitialisation, ignorez cet email</li>
                        <li>Ne partagez jamais ce lien avec qui que ce soit</li>
                    </ul>
                </div>
                
                <p>Si le bouton ne fonctionne pas, copiez et collez le lien ci-dessus dans votre navigateur.</p>
                
                <p>Cordialement,<br>
                <strong>L'équipe MOBIC GLOBAL</strong></p>
            </div>
            <div class='footer'>
                <p>© " . date('Y') . " MOBIC GLOBAL. Tous droits réservés.</p>
                <p>Cet email a été envoyé automatiquement, veuillez ne pas y répondre.</p>
            </div>
        </div>
    </body>
    </html>
    ";
    
    $headers = "MIME-Version: 1.0\r\n";
    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    $headers .= "From: MOBIC GLOBAL <no-reply@" . $_SERVER['HTTP_HOST'] . ">\r\n";
    $headers .= "Reply-To: support@" . $_SERVER['HTTP_HOST'] . "\r\n";
    $headers .= "X-Mailer: PHP/" . phpversion();
    
    return mail($to, $subject, $message, $headers);
}

/**
 * Create password reset record
 */
function createPasswordReset($conn, $email, $token) {
    $expires = date('Y-m-d H:i:s', time() + RESET_TOKEN_EXPIRY);
    
    // Invalidate previous tokens for this email
    $invalidateQuery = "UPDATE password_resets SET used = 1 WHERE email = ? AND used = 0";
    $stmt = mysqli_prepare($conn, $invalidateQuery);
    mysqli_stmt_bind_param($stmt, "s", $email);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
    
    // Create new reset record
    $query = "INSERT INTO password_resets (email, token, expires_at, created_at) 
              VALUES (?, ?, ?, NOW())";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "sss", $email, $token, $expires);
    $success = mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
    
    return $success;
}

// ============================================================================
// PROCESS FORM SUBMISSIONS
// ============================================================================

// Step 1: Request password reset (email submission)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['request_reset'])) {
    
    // Validate CSRF token
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $_SESSION['error'] = 'Session invalide. Veuillez rafraîchir la page.';
        header('Location: forgot_password.php');
        exit();
    }
    
    $email = trim($_POST['email'] ?? '');
    
    // Validate email
    if (empty($email) || !validateEmail($email)) {
        $_SESSION['error'] = 'Veuillez fournir une adresse email valide.';
        header('Location: forgot_password.php');
        exit();
    }
    
    // Check rate limiting
    $rateLimit = checkResetRateLimit($email, $conn);
    if (!$rateLimit['allowed']) {
        $_SESSION['error'] = $rateLimit['message'];
        header('Location: forgot_password.php');
        exit();
    }
    
    // Check if user exists
    $query = "SELECT user_id, username, fullname, email FROM code_users WHERE email = ? AND status = 'active'";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "s", $email);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if (mysqli_num_rows($result) === 1) {
        $user = mysqli_fetch_assoc($result);
        
        // Generate reset token
        $token = generateToken();
        
        // Create reset record
        if (createPasswordReset($conn, $email, $token)) {
            
            // Send reset email
            if (sendResetEmail($user['email'], $user['fullname'], $token)) {
                // Log successful attempt
                logResetAttempt($conn, $email, true);
                
                $_SESSION['success'] = 'Un email de réinitialisation a été envoyé à ' . htmlspecialchars($email) . '. Vérifiez votre boîte de réception (et vos spams).';
                $_SESSION['reset_email_sent'] = true;
            } else {
                $_SESSION['error'] = 'Erreur lors de l\'envoi de l\'email. Veuillez réessayer.';
            }
            
        } else {
            $_SESSION['error'] = 'Erreur lors de la création de la demande de réinitialisation.';
        }
        
    } else {
        // User not found, but show same message for security
        logResetAttempt($conn, $email, false);
        $_SESSION['success'] = 'Si un compte existe avec cet email, vous recevrez un lien de réinitialisation.';
        $_SESSION['reset_email_sent'] = true;
    }
    
    mysqli_stmt_close($stmt);
    header('Location: forgot_password.php');
    exit();
}

// Step 2: Reset password (token validation and password update)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['reset_password'])) {
    
    // Validate CSRF token
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $_SESSION['error'] = 'Session invalide. Veuillez rafraîchir la page.';
        header('Location: forgot_password.php');
        exit();
    }
    
    $token = $_POST['token'] ?? '';
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    // Validate token
    if (empty($token)) {
        $_SESSION['error'] = 'Token de réinitialisation invalide.';
        header('Location: forgot_password.php');
        exit();
    }
    
    // Validate passwords
    if (empty($password) || empty($confirm_password)) {
        $_SESSION['error'] = 'Veuillez remplir tous les champs.';
        header('Location: forgot_password.php?token=' . urlencode($token));
        exit();
    }
    
    if ($password !== $confirm_password) {
        $_SESSION['error'] = 'Les mots de passe ne correspondent pas.';
        header('Location: forgot_password.php?token=' . urlencode($token));
        exit();
    }
    
    if (strlen($password) < MIN_PASSWORD_LENGTH) {
        $_SESSION['error'] = 'Le mot de passe doit contenir au moins ' . MIN_PASSWORD_LENGTH . ' caractères.';
        header('Location: forgot_password.php?token=' . urlencode($token));
        exit();
    }
    
    // Check token validity
    $query = "SELECT pr.*, u.user_id, u.username 
              FROM password_resets pr 
              JOIN code_users u ON pr.email = u.email 
              WHERE pr.token = ? AND pr.used = 0 AND pr.expires_at > NOW() AND u.status = 'active'";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "s", $token);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if (mysqli_num_rows($result) === 1) {
        $reset = mysqli_fetch_assoc($result);
        
        // Hash new password
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        
        // Update user password
        $updateQuery = "UPDATE code_users SET password = ?, failed_attempts = 0, locked_until = NULL WHERE user_id = ?";
        $updateStmt = mysqli_prepare($conn, $updateQuery);
        mysqli_stmt_bind_param($updateStmt, "si", $hashed_password, $reset['user_id']);
        
        if (mysqli_stmt_execute($updateStmt)) {
            // Mark token as used
            $markUsedQuery = "UPDATE password_resets SET used = 1, used_at = NOW() WHERE id = ?";
            $markStmt = mysqli_prepare($conn, $markUsedQuery);
            mysqli_stmt_bind_param($markStmt, "i", $reset['id']);
            mysqli_stmt_execute($markStmt);
            mysqli_stmt_close($markStmt);
            
            // Log the password change
            $logQuery = "INSERT INTO password_change_log (user_id, changed_at, ip_address) VALUES (?, NOW(), ?)";
            $logStmt = mysqli_prepare($conn, $logQuery);
            $ip = $_SERVER['REMOTE_ADDR'];
            mysqli_stmt_bind_param($logStmt, "is", $reset['user_id'], $ip);
            mysqli_stmt_execute($logStmt);
            mysqli_stmt_close($logStmt);
            
            $_SESSION['success'] = 'Votre mot de passe a été réinitialisé avec succès. Vous pouvez maintenant vous connecter.';
            header('Location: index.php');
            exit();
            
        } else {
            $_SESSION['error'] = 'Erreur lors de la mise à jour du mot de passe.';
        }
        
        mysqli_stmt_close($updateStmt);
        
    } else {
        $_SESSION['error'] = 'Token de réinitialisation invalide ou expiré.';
    }
    
    mysqli_stmt_close($stmt);
    header('Location: forgot_password.php?token=' . urlencode($token));
    exit();
}

// ============================================================================
// DISPLAY FORMS BASED ON STEP
// ============================================================================

// Check if token is provided in URL (Step 2)
$showResetForm = false;
$resetToken = '';

if (isset($_GET['token']) && !empty($_GET['token'])) {
    $resetToken = trim($_GET['token']);
    
    // Validate token
    $query = "SELECT pr.* FROM password_resets pr 
              WHERE pr.token = ? AND pr.used = 0 AND pr.expires_at > NOW()";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "s", $resetToken);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if (mysqli_num_rows($result) === 1) {
        $showResetForm = true;
        $tokenData = mysqli_fetch_assoc($result);
    } else {
        $_SESSION['error'] = 'Token de réinitialisation invalide ou expiré.';
    }
    
    mysqli_stmt_close($stmt);
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Réinitialisation du mot de passe - MOBIC GLOBAL</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        body {
            background: linear-gradient(135deg, #84BD00 0%, #f6f4f4ff 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            padding: 20px;
        }
        .password-container {
            max-width: 500px;
            margin: 0 auto;
        }
        .reset-card {
            background: white;
            border-radius: 20px;
            padding: 40px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
        }
        .form-control {
            border-radius: 10px;
            padding: 12px 15px;
            border: 2px solid #e0e0e0;
            transition: all 0.3s;
        }
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.25rem rgba(102, 126, 234, 0.25);
        }
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            padding: 12px;
            border-radius: 10px;
            font-weight: bold;
            transition: transform 0.3s;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
        }
        .password-strength {
            height: 5px;
            margin-top: 5px;
            border-radius: 5px;
            transition: all 0.3s;
        }
        .strength-0 { background: #dc3545; width: 20%; }
        .strength-1 { background: #ffc107; width: 40%; }
        .strength-2 { background: #ffc107; width: 60%; }
        .strength-3 { background: #28a745; width: 80%; }
        .strength-4 { background: #28a745; width: 100%; }
        .requirement-list {
            list-style: none;
            padding-left: 0;
            font-size: 14px;
        }
        .requirement-list li {
            margin-bottom: 5px;
        }
        .requirement-list li.valid {
            color: #28a745;
        }
        .requirement-list li.invalid {
            color: #dc3545;
        }
        .requirement-list li i {
            margin-right: 5px;
        }
        .steps {
            display: flex;
            justify-content: space-between;
            margin-bottom: 30px;
            position: relative;
        }
        .steps:before {
            content: '';
            position: absolute;
            top: 20px;
            left: 0;
            right: 0;
            height: 2px;
            background: #e0e0e0;
            z-index: 1;
        }
        .step {
            position: relative;
            z-index: 2;
            text-align: center;
            flex: 1;
        }
        .step-circle {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #e0e0e0;
            color: #666;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 10px;
            font-weight: bold;
        }
        .step.active .step-circle {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        .step-label {
            font-size: 12px;
            color: #666;
        }
        .step.active .step-label {
            color: #667eea;
            font-weight: bold;
        }
        .back-to-login {
            text-align: center;
            margin-top: 20px;
        }
        .back-to-login a {
            color: #667eea;
            text-decoration: none;
        }
    </style>
</head>
<body>
    <div class="password-container">
        <div class="reset-card">
            <!-- Steps Indicator -->
            <div class="steps">
                <div class="step <?php echo !$showResetForm ? 'active' : ''; ?>">
                    <div class="step-circle">1</div>
                    <div class="step-label">Vérification Email</div>
                </div>
                <div class="step <?php echo $showResetForm ? 'active' : ''; ?>">
                    <div class="step-circle">2</div>
                    <div class="step-label">Nouveau Mot de Passe</div>
                </div>
            </div>

            <!-- Company Header -->
            <div class="text-center mb-4">
                <h3 style="color:#000; font-weight:bold;">MOBIC GLOBAL</h3>
                <p class="text-muted">Réinitialisation du mot de passe</p>
            </div>

            <!-- Error/Success Messages -->
            <?php if (isset($_SESSION['error'])): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-circle"></i>
                    <?php 
                    echo htmlspecialchars($_SESSION['error']); 
                    unset($_SESSION['error']);
                    ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>

            <?php if (isset($_SESSION['success'])): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="fas fa-check-circle"></i>
                    <?php 
                    echo htmlspecialchars($_SESSION['success']); 
                    unset($_SESSION['success']);
                    ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>

            <?php if ($showResetForm): ?>
                <!-- Step 2: Reset Password Form -->
                <form action="forgot_password.php" method="POST" id="resetForm">
                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                    <input type="hidden" name="token" value="<?php echo htmlspecialchars($resetToken); ?>">
                    <input type="hidden" name="reset_password" value="1">
                    
                    <div class="mb-4 text-center">
                        <i class="fas fa-key fa-3x text-primary mb-3"></i>
                        <h5>Créez votre nouveau mot de passe</h5>
                        <p class="text-muted">Votre mot de passe doit être fort et sécurisé</p>
                    </div>

                    <div class="mb-3">
                        <label for="password" class="form-label">Nouveau mot de passe</label>
                        <div class="input-group">
                            <input type="password" 
                                   name="password" 
                                   id="password"
                                   class="form-control" 
                                   placeholder="Entrez votre nouveau mot de passe" 
                                   required
                                   minlength="8"
                                   oninput="checkPasswordStrength()">
                            <button type="button" class="btn btn-outline-secondary toggle-password">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                        <div class="password-strength" id="passwordStrength"></div>
                        
                        <div class="mt-2">
                            <ul class="requirement-list" id="passwordRequirements">
                                <li id="req-length" class="invalid">
                                    <i class="fas fa-times-circle"></i> Au moins 8 caractères
                                </li>
                                <li id="req-uppercase" class="invalid">
                                    <i class="fas fa-times-circle"></i> Une lettre majuscule
                                </li>
                                <li id="req-lowercase" class="invalid">
                                    <i class="fas fa-times-circle"></i> Une lettre minuscule
                                </li>
                                <li id="req-number" class="invalid">
                                    <i class="fas fa-times-circle"></i> Un chiffre
                                </li>
                                <li id="req-special" class="invalid">
                                    <i class="fas fa-times-circle"></i> Un caractère spécial
                                </li>
                            </ul>
                        </div>
                    </div>

                    <div class="mb-4">
                        <label for="confirm_password" class="form-label">Confirmez le mot de passe</label>
                        <div class="input-group">
                            <input type="password" 
                                   name="confirm_password" 
                                   id="confirm_password"
                                   class="form-control" 
                                   placeholder="Confirmez votre nouveau mot de passe" 
                                   required
                                   minlength="8">
                            <button type="button" class="btn btn-outline-secondary toggle-password">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                        <div class="invalid-feedback" id="confirmError">
                            Les mots de passe ne correspondent pas
                        </div>
                    </div>

                    <button type="submit" class="btn btn-primary btn-block w-100 mb-3" id="resetButton">
                        <span id="buttonText">Réinitialiser le mot de passe</span>
                        <span class="spinner-border spinner-border-sm d-none" id="loadingSpinner" role="status"></span>
                    </button>
                </form>

            <?php elseif (!isset($_SESSION['reset_email_sent'])): ?>
                <!-- Step 1: Email Request Form -->
                <form action="forgot_password.php" method="POST" id="emailForm">
                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                    <input type="hidden" name="request_reset" value="1">
                    
                    <div class="mb-4 text-center">
                        <i class="fas fa-unlock-alt fa-3x text-primary mb-3"></i>
                        <h5>Mot de passe oublié ?</h5>
                        <p class="text-muted">
                            Entrez votre adresse email associée à votre compte. 
                            Nous vous enverrons un lien pour réinitialiser votre mot de passe.
                        </p>
                    </div>

                    <div class="mb-4">
                        <label for="email" class="form-label">Adresse Email</label>
                        <input type="email" 
                               name="email" 
                               id="email"
                               class="form-control" 
                               placeholder="votre@email.com" 
                               required
                               value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>">
                        <div class="form-text">
                            Assurez-vous d'entrer l'adresse email associée à votre compte MOBIC GLOBAL.
                        </div>
                    </div>

                    <button type="submit" class="btn btn-primary btn-block w-100 mb-3" id="submitButton">
                        <span id="buttonText">Envoyer le lien de réinitialisation</span>
                        <span class="spinner-border spinner-border-sm d-none" id="loadingSpinner" role="status"></span>
                    </button>
                </form>

            <?php else: ?>
                <!-- Email Sent Confirmation -->
                <div class="text-center py-4">
                    <i class="fas fa-envelope fa-4x text-success mb-4"></i>
                    <h4 class="text-success">Email envoyé !</h4>
                    <p class="text-muted">
                        Vérifiez votre boîte de réception pour le lien de réinitialisation.
                        Si vous ne voyez pas l'email, vérifiez votre dossier spam.
                    </p>
                    <div class="alert alert-info mt-3">
                        <i class="fas fa-info-circle"></i>
                        Le lien expirera dans 1 heure pour des raisons de sécurité.
                    </div>
                    <a href="forgot_password.php" class="btn btn-outline-primary mt-3">
                        <i class="fas fa-redo"></i> Renvoyer l'email
                    </a>
                </div>
                <?php unset($_SESSION['reset_email_sent']); ?>
            <?php endif; ?>

            <!-- Back to Login -->
            <div class="back-to-login">
                <a href="index.php">
                    <i class="fas fa-arrow-left"></i> Retour à la connexion
                </a>
            </div>
        </div>
    </div>

    <!-- JavaScript -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Password visibility toggle
        document.querySelectorAll('.toggle-password').forEach(button => {
            button.addEventListener('click', function() {
                const input = this.closest('.input-group').querySelector('input');
                const icon = this.querySelector('i');
                
                if (input.type === 'password') {
                    input.type = 'text';
                    icon.classList.remove('fa-eye');
                    icon.classList.add('fa-eye-slash');
                } else {
                    input.type = 'password';
                    icon.classList.remove('fa-eye-slash');
                    icon.classList.add('fa-eye');
                }
            });
        });

        // Password strength checker
        function checkPasswordStrength() {
            const password = document.getElementById('password').value;
            const strengthBar = document.getElementById('passwordStrength');
            const requirements = {
                length: password.length >= 8,
                uppercase: /[A-Z]/.test(password),
                lowercase: /[a-z]/.test(password),
                number: /[0-9]/.test(password),
                special: /[^A-Za-z0-9]/.test(password)
            };
            
            // Update requirement list
            Object.keys(requirements).forEach(req => {
                const element = document.getElementById('req-' + req);
                if (requirements[req]) {
                    element.classList.remove('invalid');
                    element.classList.add('valid');
                    element.querySelector('i').className = 'fas fa-check-circle';
                } else {
                    element.classList.remove('valid');
                    element.classList.add('invalid');
                    element.querySelector('i').className = 'fas fa-times-circle';
                }
            });
            
            // Calculate strength score
            const score = Object.values(requirements).filter(Boolean).length;
            strengthBar.className = 'password-strength strength-' + score;
            
            // Check password confirmation
            checkPasswordMatch();
        }

        // Password confirmation check
        function checkPasswordMatch() {
            const password = document.getElementById('password').value;
            const confirmPassword = document.getElementById('confirm_password').value;
            const errorElement = document.getElementById('confirmError');
            const submitButton = document.getElementById('resetButton');
            
            if (confirmPassword && password !== confirmPassword) {
                errorElement.style.display = 'block';
                submitButton.disabled = true;
                return false;
            } else {
                errorElement.style.display = 'none';
                submitButton.disabled = false;
                return true;
            }
        }

        // Form submission
        document.getElementById('resetForm')?.addEventListener('submit', function(e) {
            const password = document.getElementById('password').value;
            const confirmPassword = document.getElementById('confirm_password').value;
            
            if (password !== confirmPassword) {
                e.preventDefault();
                document.getElementById('confirmError').style.display = 'block';
                return false;
            }
            
            // Show loading state
            const button = document.getElementById('resetButton');
            const buttonText = document.getElementById('buttonText');
            const loadingSpinner = document.getElementById('loadingSpinner');
            
            button.disabled = true;
            buttonText.textContent = 'Traitement...';
            loadingSpinner.classList.remove('d-none');
            
            return true;
        });

        document.getElementById('emailForm')?.addEventListener('submit', function(e) {
            // Show loading state
            const button = document.getElementById('submitButton');
            const buttonText = document.getElementById('buttonText');
            const loadingSpinner = document.getElementById('loadingSpinner');
            
            button.disabled = true;
            buttonText.textContent = 'Envoi en cours...';
            loadingSpinner.classList.remove('d-none');
        });

        // Real-time password match check
        document.getElementById('confirm_password')?.addEventListener('input', checkPasswordMatch);
        document.getElementById('password')?.addEventListener('input', checkPasswordMatch);
        
        // Auto-check on page load
        document.addEventListener('DOMContentLoaded', function() {
            checkPasswordStrength();
            checkPasswordMatch();
        });
    </script>
</body>
</html>