<?php
// ============================================================================
// ENVIRONMENT DETECTION & CONFIGURATION
// ============================================================================

// ============================================================================
// 1. SAFE ENVIRONMENT DETECTION
// ============================================================================

// Get hostname safely
$host = $_SERVER['HTTP_HOST'] ?? 'localhost';

// Check if we're in development environment
$isLocalhost = false;

// Check for common localhost/development hostnames
$localHosts = ['localhost', '127.0.0.1', '::1', '[::1]'];

// Check exact matches
if (in_array($host, $localHosts, true)) {
    $isLocalhost = true;
} 
// Check for .local, .test, .dev domains
elseif (preg_match('/\.(local|test|dev|localhost)$/i', $host)) {
    $isLocalhost = true;
}
// Check for local IP ranges
elseif (filter_var($host, FILTER_VALIDATE_IP)) {
    // Check for private IP ranges (192.168.x.x, 10.x.x.x, 172.16-31.x.x)
    if (preg_match('/^(192\.168\.|10\.|172\.(1[6-9]|2[0-9]|3[0-1])\.)/', $host)) {
        $isLocalhost = true;
    }
}

// ============================================================================
// 2. ENVIRONMENT CONSTANTS (Define once only)
// ============================================================================

// Set environment based on detection
define('ENVIRONMENT', $isLocalhost ? 'development' : 'production');

// ============================================================================
// 3. ENVIRONMENT-SPECIFIC SETTINGS
// ============================================================================

if (ENVIRONMENT === 'production') {
    // ==============================
    // PRODUCTION SETTINGS
    // ==============================
    
    // Security
    define('FORCE_HTTPS', true);
    define('SECURE_COOKIES', true);
    // In settings/config.php
    define('DEBUG_MODE', true); // Enable for testing
    define('DISPLAY_ERRORS', true); // Show errors during testing

    // define('DEBUG_MODE', false);
    // define('DISPLAY_ERRORS', false);

    define('LOG_ERRORS', true);
    
    // URLs
    define('BASE_URL', 'https://bicnv-sa.com/MGlobal/live');
    define('SITE_URL', BASE_URL);
    
    // Database - PRODUCTION (Update these with your actual credentials)
    define('DB_HOST', 'localhost');
    define('DB_USER', 'production_username');
    define('DB_PASS', 'production_password');
    define('DB_NAME', 'production_database');
    define('DB_PORT', 3306);
    define('DB_CHARSET', 'utf8mb4');
    
    // Performance
    define('CACHE_ENABLED', true);
    define('COMPRESS_OUTPUT', true);
    
    // Email
    define('MAIL_FROM', 'noreply@mydomain.com');
    define('MAIL_FROM_NAME', 'MOBIC GLOBAL');
    
} else {
    // ==============================
    // DEVELOPMENT SETTINGS
    // ==============================
    
    // Security (relaxed for development)
    define('FORCE_HTTPS', false);
    define('SECURE_COOKIES', false);
    define('DEBUG_MODE', true);
    define('DISPLAY_ERRORS', true);
    define('LOG_ERRORS', true);
    
    // URLs - Dynamically determine base URL
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $serverName = $_SERVER['SERVER_NAME'] ?? 'localhost';
    $serverPort = $_SERVER['SERVER_PORT'] ?? 80;
    $scriptPath = dirname($_SERVER['SCRIPT_NAME'] ?? '');
    
    // Build base URL dynamically
    $baseUrl = $protocol . '://' . $serverName;
    
    // Add port if not standard
    if (($protocol === 'http' && $serverPort != 80) || 
        ($protocol === 'https' && $serverPort != 443)) {
        $baseUrl .= ':' . $serverPort;
    }
    
    // Add script path if not root
    if ($scriptPath && $scriptPath !== '/' && $scriptPath !== '\\') {
        $baseUrl .= rtrim($scriptPath, '/');
    }
    
    define('BASE_URL', $baseUrl);
    define('SITE_URL', BASE_URL);
    
    // Database - DEVELOPMENT (Update these with your local credentials)
    define('DB_HOST', 'localhost');
    define('DB_USER', 'root');
    define('DB_PASS', '');
    define('DB_NAME', 'live_2');
    define('DB_PORT', 3306);
    define('DB_CHARSET', 'utf8mb4');
    
    // Performance
    define('CACHE_ENABLED', false);
    define('COMPRESS_OUTPUT', false);
    
    // Email
    define('MAIL_FROM', 'dev@localhost');
    define('MAIL_FROM_NAME', 'MOBIC GLOBAL DEV');
}

// ============================================================================
// 4. PHP SETTINGS BASED ON ENVIRONMENT
// ============================================================================

// Error reporting
if (DEBUG_MODE) {
    error_reporting(E_ALL);
    ini_set('display_errors', '1');
    ini_set('display_startup_errors', '1');
} else {
    error_reporting(E_ALL & ~E_NOTICE & ~E_DEPRECATED & ~E_STRICT);
    ini_set('display_errors', '0');
    ini_set('display_startup_errors', '0');
}

// Log errors
ini_set('log_errors', LOG_ERRORS ? '1' : '0');
if (LOG_ERRORS) {
    ini_set('error_log', dirname(__DIR__) . '/logs/php_errors.log');
}

// Timezone
date_default_timezone_set('Africa/Bujumbura');

// ============================================================================
// 5. SECURITY HEADERS FUNCTION
// ============================================================================

/**
 * Set security headers based on environment
 */
function setSecurityHeaders() {
    // Always set these headers
    header("X-Frame-Options: DENY");
    header("X-Content-Type-Options: nosniff");
    header("X-XSS-Protection: 1; mode=block");
    header("Referrer-Policy: strict-origin-when-cross-origin");
    
    // Set CSP based on environment
    if (ENVIRONMENT === 'production') {
        header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline'; img-src 'self' data: https:;");
    } else {
        header("Content-Security-Policy: default-src 'self' 'unsafe-inline' 'unsafe-eval';");
    }
    
    // HSTS for production only (HTTPS sites)
    if (ENVIRONMENT === 'production' && FORCE_HTTPS) {
        header("Strict-Transport-Security: max-age=31536000; includeSubDomains");
    }
}

// ============================================================================
// 6. DATABASE CONNECTION FUNCTION
// ============================================================================

/**
 * Create database connection with proper error handling
 * @return mysqli|false Returns mysqli connection or false on failure
 */
function createDatabaseConnection() {
    // Create connection with error suppression
    $conn = @mysqli_connect(DB_HOST, DB_USER, DB_PASS, DB_NAME, DB_PORT);
    
    if (!$conn) {
        $error = mysqli_connect_error();
        
        // Log error
        error_log("Database connection failed: " . $error);
        
        // Return appropriate error based on environment
        if (DEBUG_MODE) {
            // Detailed error for development
            die("Database Connection Error: " . $error . 
                "<br>Host: " . DB_HOST . 
                "<br>Database: " . DB_NAME);
        } else {
            // Generic error for production
            die("System temporarily unavailable. Please try again later.");
        }
    }
    
    // Set charset
    if (!mysqli_set_charset($conn, DB_CHARSET)) {
        error_log("Failed to set charset: " . mysqli_error($conn));
    }
    
    return $conn;
}

// ============================================================================
// 7. HELPER FUNCTIONS
// ============================================================================

/**
 * Check if HTTPS should be forced and redirect if needed
 */
function enforceHTTPS() {
    if (FORCE_HTTPS && 
        (empty($_SERVER['HTTPS']) || $_SERVER['HTTPS'] === 'off') &&
        $_SERVER['REQUEST_METHOD'] === 'GET') {
        
        $redirectUrl = 'https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
        header('HTTP/1.1 301 Moved Permanently');
        header('Location: ' . $redirectUrl);
        exit();
    }
}

/**
 * Get the full URL for a path
 * @param string $path The relative path
 * @return string Full URL
 */
function url($path = '') {
    return rtrim(BASE_URL, '/') . '/' . ltrim($path, '/');
}

/**
 * Get the current page URL
 * @return string Current URL
 */
function currentUrl() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    return $protocol . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
}

/**
 * Output debug information (development only)
 * @param mixed $data Data to debug
 * @param bool $dump Use var_dump instead of print_r
 */
function debug($data, $dump = false) {
    if (!DEBUG_MODE) return;
    
    echo '<pre style="background: #f5f5f5; padding: 10px; border: 1px solid #ddd; margin: 10px;">';
    if ($dump) {
        var_dump($data);
    } else {
        print_r($data);
    }
    echo '</pre>';
}

// ============================================================================
// 8. SESSION CONFIGURATION
// ============================================================================

/**
 * Configure session settings based on environment
 */
function configureSession() {
    ini_set('session.cookie_httponly', 1);
    ini_set('session.cookie_secure', SECURE_COOKIES ? 1 : 0);
    ini_set('session.cookie_samesite', 'Strict');
    ini_set('session.use_strict_mode', 1);
    ini_set('session.use_only_cookies', 1);
    ini_set('session.gc_maxlifetime', 3600); // 1 hour
    
    // Session save path (optional)
    $sessionPath = dirname(__DIR__) . '/sessions';
    if (!is_dir($sessionPath)) {
        @mkdir($sessionPath, 0755, true);
    }
    
    if (is_writable($sessionPath)) {
        ini_set('session.save_path', $sessionPath);
    }
}

// ============================================================================
// 9. INITIALIZATION
// ============================================================================

// Configure session
configureSession();

// Set security headers
setSecurityHeaders();

// Enforce HTTPS if needed
enforceHTTPS();

// ============================================================================
// END OF CONFIGURATION
// ============================================================================