<?php
require_once 'config.php';
require_once 'auth_service.php';

class TravelInsuranceService {
    
    private $accessToken;
    
    public function __construct() {
        $this->accessToken = AuthService::getAccessToken();
    }
    
    // 1. Get Quote
    public function getQuote($userData) {
        $url = TravelInsuranceConfig::BASE_URL . TravelInsuranceConfig::ENDPOINT_QUOTE;
        
        $quoteData = [
            "context" => [
                "currency" => $userData['currency'] ?? TravelInsuranceConfig::DEFAULT_CURRENCY,
                "country" => $userData['country'] ?? TravelInsuranceConfig::DEFAULT_COUNTRY,
                "language" => $userData['language'] ?? TravelInsuranceConfig::DEFAULT_LANGUAGE
            ],
            "product_criteria" => [
                "category" => TravelInsuranceConfig::DEFAULT_PRODUCT_CATEGORY,
                "catalog" => [
                    "reference" => TravelInsuranceConfig::DEFAULT_CATALOG_REFERENCE,
                    "version" => TravelInsuranceConfig::DEFAULT_CATALOG_VERSION
                ]
            ],
            "travel" => [
                "destination_area" => $userData['destination_area'],
                "start_date" => $userData['start_date'],
                "end_date" => $userData['end_date'],
                "travelers" => [
                    "composition" => $userData['composition'],
                    "types" => [
                        "adult" => $userData['adults_count'],
                        "children" => $userData['children_count'],
                        "senior" => $userData['seniors_count']
                    ],
                    "oldest_traveler_age" => $userData['oldest_traveler_age']
                ]
            ]
        ];
        
        return $this->makeRequest($url, 'POST', $quoteData);
    }
    
    // 2. Subscribe Policy
    public function subscribePolicy($userData, $quoteCode) {
        $url = TravelInsuranceConfig::BASE_URL . TravelInsuranceConfig::ENDPOINT_POLICY;
        
        $policyData = [
            "subscription_country" => $userData['subscription_country'],
            "language_code" => $userData['language_code'],
            "quote_code" => $quoteCode,
            "agent_scope" => "",
            "policy_holder" => [
                [
                    "title" => $userData['policy_holder_title'],
                    "first_name" => $userData['policy_holder_first_name'],
                    "last_name" => $userData['policy_holder_last_name'],
                    "birth_date" => $userData['policy_holder_birth_date'],
                    "email" => $userData['policy_holder_email'],
                    "address" => $userData['policy_holder_address'],
                    "is_policy_beneficiary" => 0
                ]
            ],
            "beneficiaries" => [
                [
                    "title" => $userData['beneficiary_title'],
                    "first_name" => $userData['beneficiary_first_name'],
                    "last_name" => $userData['beneficiary_last_name'],
                    "email" => $userData['beneficiary_email'],
                    "passport_number" => $userData['beneficiary_passport_number'],
                    "phone_number" => $userData['beneficiary_phone_number'],
                    "birth_date" => $userData['beneficiary_birth_date'],
                    "address" => $userData['beneficiary_address'],
                    "passeport_exp_date" => $userData['beneficiary_passport_exp_date'],
                    "city" => $userData['beneficiary_city']
                ]
            ],
            "consents" => [],
            "payment" => [
                "type" => "MANAGED_BY_PARTNER"
            ],
            "addons" => []
        ];
        
        return $this->makeRequest($url, 'POST', $policyData);
    }
    
    // 3. Get Policy Data
    public function getPolicyData($policyId, $customerEmail) {
        $url = TravelInsuranceConfig::BASE_URL . "travel/policies/" . $policyId;
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $this->accessToken,
                'customer_email: ' . $customerEmail
            ],
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        return [
            'status_code' => $httpCode,
            'response' => json_decode($response, true)
        ];
    }
    
    // 4. Get Certificate
    public function getCertificate($policyId) {
        $url = TravelInsuranceConfig::BASE_URL . sprintf(TravelInsuranceConfig::ENDPOINT_CERTIFICATE, $policyId);
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Accept-Language: en',
                'Content-Type: application/json',
                'Authorization: Bearer ' . $this->accessToken,
                'x-quote-mode: LIVE'
            ],
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        return [
            'status_code' => $httpCode,
            'response' => json_decode($response, true)
        ];
    }
    
    // 5. Update Policy
    public function updatePolicy($policyId, $updateData) {
        $url = TravelInsuranceConfig::BASE_URL . sprintf(TravelInsuranceConfig::ENDPOINT_UPDATE_POLICY, $policyId);
        
        return $this->makeRequest($url, 'PATCH', $updateData);
    }
    
    // 6. Cancel Policy
    public function cancelPolicy($policyId, $cancellationReason) {
        $url = TravelInsuranceConfig::BASE_URL . sprintf(TravelInsuranceConfig::ENDPOINT_CANCEL_POLICY, $policyId);
        
        $cancelData = [
            "cancellation_reason" => $cancellationReason
        ];
        
        return $this->makeRequest($url, 'POST', $cancelData);
    }
    
    // Common request method
    private function makeRequest($url, $method, $data) {
        $ch = curl_init();
        
        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $this->accessToken
            ],
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30
        ];
        
        if ($method === 'POST') {
            $options[CURLOPT_POST] = true;
            $options[CURLOPT_POSTFIELDS] = json_encode($data);
        } elseif ($method === 'PATCH') {
            $options[CURLOPT_CUSTOMREQUEST] = 'PATCH';
            $options[CURLOPT_POSTFIELDS] = json_encode($data);
        }
        
        curl_setopt_array($ch, $options);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        return [
            'status_code' => $httpCode,
            'response' => json_decode($response, true)
        ];
    }
}
?>