<?php
require_once 'travel_insurance_service.php';

class TravelInsuranceRegistration {
    
    public function processRegistration($userInput) {
        $insuranceService = new TravelInsuranceService();
        
        // Step 1: Validate user input
        $validationResult = $this->validateUserInput($userInput);
        if (!$validationResult['success']) {
            return $validationResult;
        }
        
        // Step 2: Get Quote
        $quoteResult = $insuranceService->getQuote($userInput);
        if ($quoteResult['status_code'] !== 200) {
            return [
                'success' => false,
                'message' => 'Failed to get quote',
                'error' => $quoteResult['response']
            ];
        }
        
        // Extract quote code from response
        $quoteCode = $this->extractQuoteCode($quoteResult['response']);
        if (!$quoteCode) {
            return [
                'success' => false,
                'message' => 'Could not extract quote code from response'
            ];
        }
        
        // Step 3: Subscribe Policy
        $policyResult = $insuranceService->subscribePolicy($userInput, $quoteCode);
        if ($policyResult['status_code'] !== 200) {
            return [
                'success' => false,
                'message' => 'Failed to subscribe policy',
                'error' => $policyResult['response']
            ];
        }
        
        // Extract policy ID from response
        $policyId = $this->extractPolicyId($policyResult['response']);
        
        // Step 4: Get Certificate (optional)
        $certificateResult = $insuranceService->getCertificate($policyId);
        
        return [
            'success' => true,
            'message' => 'Travel insurance registered successfully',
            'data' => [
                'policy_id' => $policyId,
                'quote_code' => $quoteCode,
                'policy_details' => $policyResult['response'],
                'certificate' => $certificateResult['response']
            ]
        ];
    }
    
    private function validateUserInput($input) {
        $requiredFields = [
            'destination_area', 'start_date', 'end_date', 'composition',
            'adults_count', 'children_count', 'seniors_count', 'oldest_traveler_age',
            'policy_holder_title', 'policy_holder_first_name', 'policy_holder_last_name',
            'policy_holder_birth_date', 'policy_holder_email', 'policy_holder_address',
            'beneficiary_title', 'beneficiary_first_name', 'beneficiary_last_name',
            'beneficiary_email', 'beneficiary_passport_number', 'beneficiary_phone_number',
            'beneficiary_birth_date', 'beneficiary_address', 'beneficiary_passport_exp_date',
            'beneficiary_city'
        ];
        
        foreach ($requiredFields as $field) {
            if (empty($input[$field])) {
                return [
                    'success' => false,
                    'message' => "Missing required field: $field"
                ];
            }
        }
        
        // Validate dates
        if (strtotime($input['start_date']) >= strtotime($input['end_date'])) {
            return [
                'success' => false,
                'message' => 'End date must be after start date'
            ];
        }
        
        // Validate email
        if (!filter_var($input['policy_holder_email'], FILTER_VALIDATE_EMAIL)) {
            return [
                'success' => false,
                'message' => 'Invalid email address'
            ];
        }
        
        return ['success' => true];
    }
    
    private function extractQuoteCode($response) {
        // Implementation depends on the actual response structure
        return $response['quote_code'] ?? null;
    }
    
    private function extractPolicyId($response) {
        // Implementation depends on the actual response structure
        return $response['policy_id'] ?? null;
    }
}

// Example usage from Android app
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $userInput = json_decode(file_get_contents('php://input'), true);
    
    $registration = new TravelInsuranceRegistration();
    $result = $registration->processRegistration($userInput);
    
    header('Content-Type: application/json');
    echo json_encode($result);
}
?>