<?php
require_once('../fpdfLib/fpdf.php');

class InsurancePDFGeneratorPBMOAPK {
    private $pdf;
    private $data;
    private $headerImage = 'ressources/img/header1s.png';
    private $stampImage = 'ressources/img/cachervs.png'; // New stamp image
    private $generalFontSize = 9;

    public function __construct($data) {
        $this->data = $data;
        $this->pdf = new FPDF('P', 'mm', 'A4'); // A4 paper
        $this->pdf->SetMargins(10, 10, 10);
        $this->pdf->AddPage();
        $this->pdf->SetFont('Arial', '', 8);
    }

    public function generate() {
        $this->addHeader();
        $this->addPolicyInfo();
        $this->addGoodsCharacteristics();
        $this->addInsuranceDetails();
        $this->addGeneralCondition();    // <-- NEW insertion
        $this->addSignatures();
        return $this->pdf->Output('S');
    }

    /** HEADER: IMAGE + TITLE **/
    private function addHeader() {
        if (file_exists($this->headerImage)) {
            $this->pdf->Image($this->headerImage, 10, 3, 150); // full width
            $this->pdf->Ln(35); // Space after image
        }
        $this->pdf->SetFont('Arial', 'B', 8);
        $this->pdf->Cell(0, 10, 'CONDITIONS PARTICULIERES: ASSURANCE AUTOMOBILE', 0, 1, 'C');
        $this->addHorizontalLine();
    }

    /** POLICY INFO **/
    private function addPolicyInfo() {
        $this->pdf->SetFont('Arial', '', 8);

        $this->addRow('Date de l\'operation:', $this->formatDate($this->data['DateEmission']), 'Intermediaire:', 'N/A');
        $this->addRow('Numero de Police:', $this->data['Numpol'], 'Facture No.:', $this->data['facture']);
        $this->addHorizontalLine();

        $this->addRow('Date d\'effet:', $this->formatDate($this->data['DateEffet']), 'Date d\'echeance:', $this->formatDate($this->data['DateEcheance']));
        $this->addHorizontalLine();

        $this->addRow('Preneur d\'assurance:', $this->data['Proprietaire'], 'Residence:', $this->data['residence']);
        $this->addRow('NIF:', $this->data['NIF'], 'Souscripteur No.:', $this->data['user_code']);
        $this->addRow('Adresse:', $this->data['residence'], 'Telephone:', $this->data['phone']);
    }

    /** VEHICLE INFO **/
    private function addGoodsCharacteristics() {
        $this->addHorizontalLine();
        $this->pdf->SetFont('Arial', 'B', 8);
        $this->pdf->Cell(0, 10, 'CARACTERISTIQUES DES OBJETS ASSURES', 0, 1, 'C');
        $this->addHorizontalLine();
        $this->pdf->SetFont('Arial', '', 8);

        $this->addRow('Immatric. Veh.:', $this->data['Plaque'], 'No Chassis:', $this->data['NumChasis']);
        $this->addRow('Marque:', $this->data['ClassMarkModel'], 'Puissance:', $this->data['PuissanceCV']);
        $this->addRow('Pass. en Cabine:', $this->data['PaleSeatCabine'], 'Pass. sur plateau:', $this->data['SeatPlacePlato']);
        $this->addRow('Usage:', $this->data['UsageAuto'], 'No. Certificat.:', $this->data['NumCertificat']);
    }

    /** INSURANCE DETAILS **/
    private function addInsuranceDetails() {
        $this->addHorizontalLine();
        $this->pdf->SetFont('Arial', 'B', 8);
        $this->pdf->Cell(0, 10, 'DETAILS DE LA PRIME D\'ASSURANCE AUTOMOBILE', 0, 1, 'C');
        $this->addHorizontalLine();
        $this->pdf->SetFont('Arial', 'B', 8);

        $this->pdf->Cell(45, 8, 'Objet assure', 0, 0);
        $this->pdf->Cell(70, 8, 'Garantie', 0, 0);
        $this->pdf->Cell(40, 8, 'Prime Nette', 0, 0);
        $this->pdf->Cell(0, 8, 'Prime Brute', 0, 1);

        $this->pdf->SetFont('Arial', '', 8);
        $this->pdf->Cell(45, 8, $this->data['Plaque'], 0, 0);
        $this->pdf->Cell(70, 8, $this->data['Garantie'], 0, 0);
        $this->pdf->Cell(40, 8, $this->formatNumber($this->data['PN']), 0, 0);
        $this->pdf->Cell(0, 8, $this->formatNumber($this->data['PB']), 0, 1);

        $this->addHorizontalLine();
        
        $this->addRow('PN ET FRAIS HORS TAXE:', $this->formatNumber($this->data['PN']));

        $this->addHorizontalLine();
        $this->addRow('Prime nette:', $this->formatNumber($this->data['PN']), 'Frais admin.:', $this->formatNumber($this->data['FRAIS']));
        $this->addRow('Autres frais (CD):', $this->formatNumber($this->data['CD']), 'TVA:', $this->formatNumber($this->data['TVA']));
        $this->addRow('Rabbais:', '0', 'Modalite de Paiement:', $this->data['mode_payment']);
    }
    
    /**
 * Adds the 'Conditions Générales' section as structured text.
 * Skips original PDF header/footer, focuses on article text.
 */
    private function addGeneralCondition() {
        // Start on a new page
        $this->pdf->AddPage();
        $this->pdf->SetFont('Arial', 'B', 8);
        $this->pdf->Cell(0, 10, 'CONDITIONS GENERALES DE L-ASSURANCE AUTOMOBILE', 0, 1, 'C');
        $this->addHorizontalLine();
    
        // Set font for text
        $this->pdf->SetFont('Arial', '', $this->generalFontSize);
        $this->pdf->Ln(2);
    
        // Insert each Article / Section
        $sections = [
            'Definitions' => "Accessoire: Tout element d-enjolivement ou d-amelioration fixe sur votre vehicule.\nAccident:Tout evenement non intentionnel de l-assure entrainant des dommages corporels ou materiels et provenant de l-action violente, soudaine et imprevisible d-une cause exterieure au vehicule.\nAffaire : Saisine d-une juridiction par des parties qui s-opposent sur des memes faits, afin que leurs positions soient tranchees.\nAgression : Atteinte physique ou morale a la personne assuree\nAmenagement : Tout element de modification ou de transformation de votre vehicule fixe a celui-ci.\nAntecedents : Informations relatives au « passe automobile » du souscripteur d-assurance, du proprietaire du vehicule, du (ou des) conducteur (s) designe (s) aux conditions particulieres.\nAvocat postulant : Avocat qui represente une partie devant un tribunal de Grande Instance ou une Cour d-Appel lorsque l-Avocat choisi par le client pour plaider son dossier n-est pas inscrit au barreau devant lequel le proces a lieu.\nConducteur principal : La personne physique parcourant chaque annee le plus grand nombre de kilometres en tant que conducteur du vehicule assure.\nConducteur occasionnel : Tout autre conducteur.\nConvention d-honoraires : Convention signee entre l-Avocat et son client fixant les honoraires et les modalites de reglement.\nCreance : Droit dont vous disposez pour exiger d-un tiers la remise d-une somme d-argent.\nCrevaison : Par crevaison, il faut entendre tout echappement d-air (degonflement ou eclatement d-un pneumatique, qui rend impossible l-utilisation du vehicule dans des conditions normales de securite et ayant pour effet d-immobiliser le vehicule sur le lieu de l-incident et de necessiter un depannage ou remorquage dans un garage pour y effectuer les reparations necessaires.\nDecheance : Lorsque vous ne respectez pas les obligations auxquelles vous etes tenu par ce contrat, vous pouvez perdre tout ou partie du droit a indemnite ou meme nous rembourser une indemnite reglee e un tiers.\nDepens : Part des frais engendres par un proces dont le montant fait l-objet d-une tarification par voie reglementaire
(notamment emoluments des officiers publics ou ministeriels, indemnites des temoins), ou par decision judiciaire (remuneration des experts et techniciens) et que le tribunal met à la charge de l-une des parties qui est le plus souvent la perdante.\nDol : Manoeuvres, mensonges, silence sur une information (réticence dolosive) ayant pour objet de tromper l-une des parties en vue d-obtenir son consentement.\nDomicile principal : Lieu de résidence principal et habituel du beneficiaire figurant comme domicile sur la declaration d-
impots sur le revenu. Il est situe au Burundi."
            
            
            
            
            // 'Article 1 – Definitions' => "Definitions completes\n Accessoire\n: Tout élément d’enjolivement ou d’amelioration fixé sur votre véhicule.",
            // 'Article 2 – Bien assuré' => "Le bien assuré désigne: le véhicule terrestre à moteur…",
            // 'Article 3 – Garantie responsabilité civile' => "Responsabilité civile: couvre les dommages corporels…",
            // 'Article 4 – Défense & Recours' => "Défense pénale: l’assureur garantit votre défense en cas de poursuite…",
            // 'Article 5 – Exclusions générales' => "Exclusions: ce qui n'est pas couvert, y compris usure normale…",
            // 'Article 6 – Franchise et modalités' => "Franchises: montants à la charge de l’assuré…",
            // 'Article 7 – Paiement des primes' => "Modalités de paiement, échéances, refus et pénalités…",
            // 'Article 8 – Résiliation et modification' => "Conditions de résiliation, préavis, changement de risque…",
            // 'Article 9 – Prescription' => "Toute action dérivée du contrat est prescrite après 2 ans…",
            // 'Article 10 – Dispositions diverses' => "Loi applicable, notifications, force majeure, etc."
        ];
    
        foreach ($sections as $heading => $text) {
            $this->pdf->SetFont('Arial', 'B', $this->generalFontSize);
            $this->pdf->Cell(0, 6, $heading, 0, 1);
            $this->pdf->Ln(1);
            $this->pdf->SetFont('Arial', '', $this->generalFontSize);
            $this->pdf->MultiCell(0, 5, $text, 0, 'J');
            $this->pdf->Ln(5);
        }
    }

    /** SIGNATURES **/
    private function addSignatures() {
    $this->addHorizontalLine();
    $this->pdf->SetFont('Arial', 'B', 9);
    $this->pdf->Cell(12, 8, 'N.B.*:', 0, 0);
    $this->pdf->Cell(0, 8, 'La presente police prend effet dans un delai de 6h ou le lendemain a 00h00.', 0, 1);
    $this->addHorizontalLine();
    $this->pdf->SetFont('Arial', '', 9);
    $this->pdf->MultiCell(0, 8, 'Le preneur d\'assurance declare avoir pris connaissance des conditions generales et particulieres qui font partie integrante de la police d\'assurance.', 0, 1);
    $this->pdf->Ln(5);

    $this->pdf->Cell(0, 8, 'Fait a Bujumbura, le ' . $this->formatDate($this->data['DateEmission']), 0, 1, 'C');
    $this->pdf->Ln(8);

    $this->pdf->Cell(95, 8, 'LE PRENEUR', 0, 0, 'C');
    $this->pdf->Cell(95, 8, 'POUR LA COMPAGNIE D\'ASSURANCE', 0, 1, 'C');
    $this->pdf->Ln(10);

    // Capture current Y before printing names
    $signatureY = $this->pdf->GetY();
    
    $this->pdf->Cell(95, 8, $this->data['Proprietaire'], 0, 0, 'C');
    $this->pdf->Cell(95, 8, $this->data['Creator'], 0, 1, 'C');

    // Add the stamp image slightly above the right signature
    if (file_exists($this->stampImage)) {
        $this->pdf->Image($this->stampImage, 130, $signatureY - 15, 40); 
        // X=140mm, Y slightly above signature line, Width=50mm
    }
}
   /* private function addSignatures() {
        $this->addHorizontalLine();
        $this->pdf->SetFont('Arial', 'B', 9);
        $this->pdf->Cell(12, 8, 'N.B.*:', 0, 0);
        $this->pdf->Cell(0, 8, 'La presente police prend effet dans un delai de 6h ou le lendemain a 00h00.', 0, 1);
        $this->addHorizontalLine();
        $this->pdf->SetFont('Arial', '', 9);
        $this->pdf->MultiCell(0, 8, 'Le preneur d\'assurance declare avoir pris connaissance des conditions generales et particulieres qui font partie integrante de la police d\'assurance.', 0, 1);
        $this->pdf->Ln(5);

        $this->pdf->Cell(0, 8, 'Fait a Bujumbura, le ' . $this->formatDate($this->data['DateEmission']), 0, 1, 'C');
        $this->pdf->Ln(3);

        $this->pdf->Cell(95, 8, 'LE PRENEUR', 0, 0, 'C');
        $this->pdf->Cell(95, 8, 'POUR LA COMPAGNIE D\'ASSURANCE', 0, 1, 'C');
        $this->pdf->Ln(5);

        $this->pdf->Cell(95, 8, $this->data['Proprietaire'], 0, 0, 'C');
        $this->pdf->Cell(95, 8, $this->data['Creator'], 0, 1, 'C');
    }*/

    /** HELPERS **/
    private function addRow($label1, $value1, $label2 = '', $value2 = '') {
        $this->pdf->SetFont('Arial', '', 9);
        $this->pdf->Cell(50, 5, $label1, 0, 0);
        $this->pdf->Cell(50, 5, $value1, 0, 0);
        if ($label2) {
            $this->pdf->Cell(40, 5, $label2, 0, 0);
            $this->pdf->Cell(0, 5, $value2, 0, 1);
        } else {
            $this->pdf->Cell(0, 5, '', 0, 1);
        }
    }

    private function addHorizontalLine($width = 0.3, $color = [150, 150, 150], $padding = 5) {
        $this->pdf->SetDrawColor($color[0], $color[1], $color[2]);
        $this->pdf->SetLineWidth($width);
        $y = $this->pdf->GetY();
        $this->pdf->Line(10, $y + ($padding / 2), 200, $y + ($padding / 2));
        $this->pdf->Ln($padding);
    }

    private function formatDate($dateString) {
        return !empty($dateString) ? date('d/m/Y', strtotime($dateString)) : '';
    }
    private function formatNumber($number) {
        return number_format($number, 2, ',', ' ');
    }
}
