<?php
session_start();
require_once('../includes/DbOperations.php');

class SoftInvoiceSender
{
    private $db;
    private $contractId;
    private $invoiceDetails;
    private $response = ['error' => false, 'message' => '', 'result' => []];
    
    const DEFAULT_ITEM_QUANTITY = '1.0';
    const DEFAULT_ITEM_CT = '0.0';
    const DEFAULT_ITEM_TL = '0.0';
    const DEVICE_ID = 'wsl400082911100284';
    const INVOICE_TYPE = 'FN';
    const PAYMENT_TYPE = '1';
    const INVOICE_CURRENCY = 'BIF';
    
    // Taxpayer information constants
    const TP_NAME = 'BUSINESS INSURANCE AND REINSURANCE COMPANY';
    const TP_TIN = '4000829111';
    const TP_TRADE_NUMBER = '08794';
    const TP_POSTAL_NUMBER = '362';
    const TP_PHONE_NUMBER = '22280042';
    const TP_ADDRESS_PROVINCE = 'BUJUMBURA MAIRIE';
    const TP_ADDRESS_COMMUNE = 'MUKAZA';
    const TP_ADDRESS_QUARTIER = 'ROHERO I';
    const TP_ADDRESS_AVENUE = 'ONU';
    const TP_ADDRESS_NUMBER = '6';
    const TP_FISCAL_CENTER = 'DGC';
    const TP_ACTIVITY_SECTOR = 'ASSURANCE';
    const TP_LEGAL_FORM = 'SA';
    
    public function __construct()
    {
        $this->validateRequest();
        $this->db = new DbOperations();
        $this->contractId = $_POST['idProprio'];
    }
    
    public function process()
    {
        try {
            $this->validateContract();
            $this->loadInvoiceDetails();
            $this->prepareInvoiceData();
            $this->sendToEBMS();
            $this->updateDatabase();
            $this->redirectWithSuccess();
        } catch (Exception $e) {
            $this->handleError($e->getMessage(), $e->getCode());
        }
    }
    
    private function validateRequest()
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->sendJsonResponse('Invalid request method', 405);
        }
        
        if (!isset($_POST['btnPostDataEBMSEndPoint'])) {
            $this->sendJsonResponse('Required field(s) are missing', 400);
        }
    }
    
    private function validateContract()
    {
        if (!$this->db->PullDetailsSoftInvoice($this->contractId)) {
            $this->sendJsonResponse('Invalid Proprio', 404);
        }
    }
    
    private function loadInvoiceDetails()
    {
        $this->invoiceDetails = $this->db->getDetailsSoftInvoiceById($this->contractId);
        
        if (empty($this->invoiceDetails)) {
            throw new Exception('Invoice details not found', 404);
        }
    }
    
    private function prepareInvoiceData()
    {
        $this->prepareSignature();
        $this->prepareInvoiceItems();
        $this->prepareCustomerData();
        $this->prepareTaxpayerData();
    }
    
    private function prepareSignature()
    {
        $invoiceDate = $this->invoiceDetails['invoice_signature_date'] ?? null;
        
        if (empty($invoiceDate)) {
            $invoiceOriginalDate = $this->invoiceDetails['invoice_date'];
            $dateSignature = date("YmdHis", strtotime($invoiceOriginalDate));
            
            $this->invoiceDetails['signature'] = self::TP_TIN . '/' . self::DEVICE_ID . '/' . 
                                                $dateSignature . '/' . $this->invoiceDetails['invoice_number'];
            $this->invoiceDetails['invoice_original_date'] = $invoiceOriginalDate;
            $this->invoiceDetails['invoice_original_date_v1'] = null;
        } else {
            $this->invoiceDetails['invoice_original_date'] = $invoiceDate;
            $this->invoiceDetails['invoice_original_date_v1'] = $invoiceDate;
            $this->invoiceDetails['signature'] = $this->invoiceDetails['invoice_signature'];
        }
    }
    
    private function prepareInvoiceItems()
    {
        $itemPrice = $this->invoiceDetails['PN'] + 
                     $this->invoiceDetails['FRAIS'] + 
                     $this->invoiceDetails['CD'];
        
        // Calculate values
        $itemPriceFinal = (float) $itemPrice;
        $itemPriceNtva = (float) $itemPrice;
        $tvaAmount = (float) $this->invoiceDetails['TVA'];
        $itemPriceWvat = (float) $this->invoiceDetails['PB'];
        $itemTotalAmount = (float) $this->invoiceDetails['PB'];
        $itemName = $this->sanitizeInput($this->invoiceDetails['item_designation']);
        
        // Store these for later use in database update
        $this->invoiceDetails['calculated_values'] = [
            'item_price_final' => $itemPriceFinal,
            'item_price_ntva' => $itemPriceNtva,
            'tva_amount' => $tvaAmount,
            'item_price_wvat' => $itemPriceWvat,
            'item_total_amount' => $itemTotalAmount,
            'item_name' => $itemName
        ];
        
        $this->response['result'] = [
            'invoice_number' => $this->invoiceDetails['invoice_number'],
            'invoice_date' => $this->invoiceDetails['invoice_original_date'],
            'invoice_type' => self::INVOICE_TYPE,
            'invoice_signature' => $this->invoiceDetails['signature'],
            'invoice_signature_date' => $this->invoiceDetails['invoice_original_date_v1'],
            'invoice_items' => [
                $this->createInvoiceItem($itemPriceFinal, $itemPriceNtva, $tvaAmount, $itemPriceWvat, $itemTotalAmount, $itemName)
            ]
        ];
    }
    
    private function createInvoiceItem($itemPriceFinal, $itemPriceNtva, $tvaAmount, $itemPriceWvat, $itemTotalAmount, $itemName)
    {
        return [
            'device_id' => self::DEVICE_ID,
            'item_designation' => $itemName,
            'item_quantity' => self::DEFAULT_ITEM_QUANTITY,
            'item_price' => $itemPriceFinal,
            'item_ct' => self::DEFAULT_ITEM_CT,
            'item_tl' => self::DEFAULT_ITEM_TL,
            'item_price_nvat' => $itemPriceNtva,
            'vat' => $tvaAmount,
            'item_price_wvat' => $itemPriceWvat,
            'item_total_amount' => $itemTotalAmount
        ];
    }
    
    private function prepareCustomerData()
    {
        $this->response['result'] += [
            'customer_name' => $this->invoiceDetails['customer_name'],
            'customer_TIN' => '',
            'customer_address' => $this->invoiceDetails['customer_address'],
            'vat_customer_payer' => '0'
        ];
    }
    
    private function prepareTaxpayerData()
    {
        $this->response['result'] += [
            'tp_type' => '1',
            'tp_name' => self::TP_NAME,
            'tp_TIN' => self::TP_TIN,
            'tp_trade_number' => self::TP_TRADE_NUMBER,
            'tp_postal_number' => self::TP_POSTAL_NUMBER,
            'tp_phone_number' => self::TP_PHONE_NUMBER,
            'tp_address_province' => self::TP_ADDRESS_PROVINCE,
            'tp_address_commune' => self::TP_ADDRESS_COMMUNE,
            'tp_address_quartier' => self::TP_ADDRESS_QUARTIER,
            'tp_address_avenue' => self::TP_ADDRESS_AVENUE,
            'tp_address_number' => self::TP_ADDRESS_NUMBER,
            'vat_taxpayer' => '1',
            'ct_taxpayer' => '0',
            'tl_taxpayer' => '0',
            'tp_fiscal_center' => self::TP_FISCAL_CENTER,
            'tp_activity_sector' => self::TP_ACTIVITY_SECTOR,
            'tp_legal_form' => self::TP_LEGAL_FORM,
            'payment_type' => self::PAYMENT_TYPE,
            'invoice_currency' => self::INVOICE_CURRENCY,
            'cancelled_invoice_ref' => '',
            'invoice_ref' => '',
            'cn_motif' => ''
        ];
    }
    
    private function sendToEBMS()
    {
        $token = $this->db->getTokenByRequest();
        
        if (empty($token)) {
            $this->sendJsonResponse('Invalid Token', 401);
        }
        
        $response = $this->makeHttpRequest(
            'https://ebms.obr.gov.bi:8443/ebms_api/addInvoice/',
            $this->response['result'],
            $token
        );
        
        if ($response['status'] !== 200) {
            throw new Exception($response['body'], $response['status']);
        }
        
        // Store the API response for potential debugging
        $this->invoiceDetails['api_response'] = $response;
    }
    
    private function makeHttpRequest($url, $data, $token)
    {
        $headers = [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $token
        ];
        
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30
        ]);
        
        $body = curl_exec($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if (curl_errno($ch)) {
            throw new Exception('cURL error: ' . curl_error($ch), 500);
        }
        
        curl_close($ch);
        
        return ['status' => $status, 'body' => $body];
    }
    
    private function updateDatabase()
    {
        // Update soft contract
        if ($this->db->getSignContractSoftMain($this->contractId) !== 1) {
            throw new Exception('Failed to update soft contract', 500);
        }
        
        // Prepare parameters for OBR details
        $calculated = $this->invoiceDetails['calculated_values'];
        
        // Save OBR details
        $result = $this->db->getOBRDetailsSoftAdded(
            $this->invoiceDetails['invoice_number'],                    // invoice_number
            $this->invoiceDetails['invoice_original_date'],            // invoiceOriginal_date
            $this->invoiceDetails['customer_name'],                    // customer_name
            $this->invoiceDetails['customer_address'],                 // customer_address
            $this->invoiceDetails['signature'],                        // signature
            $this->invoiceDetails['invoice_original_date_v1'],         // invoiceOriginal_date_v1
            self::DEVICE_ID,                                           // device_id
            $calculated['item_name'],                                  // item_name
            $calculated['item_price_final'],                           // item_price_final
            $calculated['item_price_ntva'],                            // item_price_ntva
            $calculated['tva_amount'],                                 // tva_amount
            $calculated['item_price_wvat'],                            // item_price_wvat
            $calculated['item_total_amount'],                          // item_total_amount
            $this->contractId,                                         // contract_Id
            $this->invoiceDetails['agence'] ?? '',                     // ownership_agence
            $this->invoiceDetails['produit'] ?? ''                     // product
        );
        
        $this->handleOBRResult($result);
    }
    
    private function handleOBRResult($result)
    {
        switch ($result) {
            case 0:
                throw new Exception('Invoice ID already exists in saving table list', 409);
            case 1:
                // Success - continue processing
                break;
            case 2:
                throw new Exception('Some error occurred while saving OBR details', 500);
            default:
                throw new Exception('Unexpected result from OBR details save operation', 500);
        }
    }
    
    private function redirectWithSuccess()
    {
        $apiStatus = $this->invoiceDetails['api_response']['status'] ?? 'N/A';
        $apiResult = $this->invoiceDetails['api_response']['body'] ?? 'N/A';
        
        $this->response['message'] = "Invoice sent and Signature saved <br><br>" . 
                                     $apiStatus . "--->" . $apiResult . "<br><br>";
        $_SESSION['success'] = $this->response['message'];
        header('Location: ../../syst_tax_soft_transfert_control');
        exit;
    }
    
    private function handleError($message, $statusCode = 500)
    {
        $_SESSION['stata'] = $statusCode . ' -> ' . $message;
        header('Location: ../../syst_tax_soft_transfert_control');
        exit;
    }
    
    private function sendJsonResponse($message, $statusCode)
    {
        http_response_code($statusCode);
        echo json_encode(['error' => $message]);
        exit;
    }
    
    private function sanitizeInput($input)
    {
        $input = trim($input);
        $input = stripslashes($input);
        $input = htmlspecialchars($input);
        return $input;
    }
}

// Execute the process
$invoiceSender = new SoftInvoiceSender();
$invoiceSender->process();
?>

<?php
/*session_start();
require_once('../includes/DbOperations.php');
$resp_data = array();
$Sign = 2;
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
   if (!isset($_POST['btnPostDataEBMSEndPoint'])) {
      http_response_code(400); // Bad Request
      $resp_data['error'] = 'Required field(s) are missing';
      echo json_encode($resp_data);
      return; } 
  
      $contract_Id = $_POST['idProprio'];
      $db = new DbOperations();

      if (!$db->PullDetailsSoftInvoice($contract_Id)) {
         http_response_code(404); // Not Found
         $resp_data['error'] = 'Invalid Proprio';
         echo json_encode($resp_data);
         return;} 

         $invoiceDetails        = $db->getDetailsSoftInvoiceById($contract_Idd);
         $invoice_number        = $invoiceDetails['invoice_number'];

         $invoice_date          = $invoiceDetails['invoice_signature_date'] ?? null;
        if (empty($invoice_date)) {
                // 🔴 Case: NULL, empty string, or not set
                // Do another job here
         $invoiceOriginal_date      = $invoiceDetails['invoice_date'];
         $date_signature            = date("YmdHis",strtotime($invoiceOriginal_date));
         $signature                 = '4000829111/wsl400082911100284/' . $date_signature . '/' . $invoice_number;
   

        } else {
                // 🟢 Case: value exists
                // Normal processing
        $invoiceOriginal_date       = $invoice_date;
        $signature                  = $invoiceDetails['invoice_signature'];
        $invoiceOriginal_date_v1       = $invoiceOriginal_date;

        }


         $customer_name         = $invoiceDetails['customer_name'];
         $customer_address      = $invoiceDetails['customer_address'];
         $ownership_agence      = $invoiceDetails['agence'];
         $product               = $invoiceDetails['produit'];
         $device_id             = "wsl400082911100284";
         define('DEFAULT_ITEM_QUANTITY', '1.0');
         define('DEFAULT_ITEM_CT', '0.0');
         define('DEFAULT_ITEM_TL', '0.0');
         $item_designation      = $invoiceDetails['item_designation'];
         $invoice_PN            = $invoiceDetails['PN'];
         $invoice_FRAIS         = $invoiceDetails['FRAIS'];
         $invoice_CD            = $invoiceDetails['CD'];
         

         $item_price            = $invoice_PN + $invoice_FRAIS + $invoice_CD;
         $item_price_nvat       =  $item_price;
         $invoice_TVA           = $invoiceDetails['TVA'];
         $itemTotValu           = $invoiceDetails['PB'];
         $item_total_amountPB   = $itemTotValu;

          function sanitize_input($input) {
               $input = trim($input);
               $input = stripslashes($input);
               $input = htmlspecialchars($input);
               return $input;
            }
            $item_name = sanitize_input($item_designation);
            $item_price_final = floatval($item_price);
            $item_price_ntva = floatval($item_price_nvat);
            $tva_amount = floatval($invoice_TVA);
            $item_price_wvat = floatval($itemTotValu);
            $item_total_amount = floatval($item_total_amountPB);

            $token = $db->getTokenByRequest();
            if(empty($token)) {
                  http_response_code(401); 
                  $resp_data['error'] = 'Invalid Token';
                  echo json_encode($resp_data['result']);
                  return; }

        $url = 'https://ebms.obr.gov.bi:8443/ebms_api/addInvoice/';

        $tokenTwo = $token;

        $resp_data['result'] = array(
                  "invoice_number" => $invoice_number,
                  "invoice_date" => $invoiceOriginal_date,
                  "invoice_type" => "FN",
                  "tp_type" => "1",
                  "tp_name" => "BUSINESS INSURANCE AND REINSURANCE COMPANY",
                  "tp_TIN" => "4000829111",
                  "tp_trade_number" => "08794",
                  "tp_postal_number" => "362",
                  "tp_phone_number" => "22280042",
                  "tp_address_province" => "BUJUMBURA MAIRIE",
                  "tp_address_commune" => "MUKAZA",
                  "tp_address_quartier" => "ROHERO I",
                  "tp_address_avenue" => "ONU",
                  "tp_address_number" => "6",
                  "vat_taxpayer" => "1",
                  "ct_taxpayer" => "0",
                  "tl_taxpayer" => "0",
                  "tp_fiscal_center" => "DGC",
                  "tp_activity_sector" => "ASSURANCE",
                  "tp_legal_form" => "SA",
                  "payment_type" => "1",
                  "invoice_currency" => "BIF",
                  "customer_name" => $customer_name,
                  "customer_TIN" => "",
                  "customer_address" => $customer_address,
                  "vat_customer_payer" => "0",
                  "cancelled_invoice_ref" => "",
                  "invoice_ref" => "",
                  "cn_motif" => "",
                  "invoice_signature" => $signature,
                  "invoice_signature_date" => $invoiceOriginal_date_v1,
                  "invoice_items" => array(
                     array(
                        "device_id" => $device_id,
                        "item_designation" => $item_name,
                        "item_quantity" => DEFAULT_ITEM_QUANTITY,
                        "item_price" => $item_price_final,
                        "item_ct" => DEFAULT_ITEM_CT,
                        "item_tl" => DEFAULT_ITEM_TL,
                        "item_price_nvat" => $item_price_ntva,
                        "vat" => $tva_amount,
                        "item_price_wvat" => $item_price_wvat,
                        "item_total_amount" => $item_total_amount
                     )
                  )
               );

            $headers = array(
                  'Content-Type: application/json',
                  'Authorization: Bearer ' . $tokenTwo
              );
              $data = json_encode($resp_data['result']);
              $ch = curl_init();
              curl_setopt($ch, CURLOPT_URL, $url);
              curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
              curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
              curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
              $result = curl_exec($ch);
              $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
              curl_close($ch);

               if($status==200) {
                // Notify Contract
               $feedBack = $db->getSignContractSoftMain($contract_Id);
               if ( $feedBack == 1 ) {
               
                    $finalproof = $db->getOBRDetailsAdded($invoice_number,$invoiceOriginal_date, $customer_name, $customer_address, 
                    $signature, $invoiceOriginal_date_v1, $device_id, $item_name, $item_price_final, $item_price_ntva, $tva_amount, 
                    $item_price_wvat, $item_total_amount, $contract_Id, $ownership_agence, $product);
                            if ($finalproof == 1) {
                                $resp_data['error'] = false;
                                $resp_data['message'] = "Invoice sent and Signature saved <br><br>".$status."--->".$result."<br><br>";
                                $_SESSION['success'] = $resp_data['message'];
                                header('Location: ../../syst_tax_soft_transfert_control');
                                exit;
                            }elseif($finalproof == 2){
                                $resp_data['error'] = true;
                                $resp_data['message'] = "Invoice sent but Some error occurred for saving please try again";
                                $_SESSION['stata'] = $resp_data['message'];
                                header('Location: ../../syst_tax_soft_transfert_control');
                                exit;
                            }elseif($finalproof == 0){
                                $resp_data['error'] = true;
                                $resp_data['message'] = "Invoice sent but Invoice Id exist in saving table list";
                                $_SESSION['success'] = $resp_data['message'];
                                header('Location: ../../syst_tax_soft_transfert_control');
                                exit;
                            }

                 }else if ( $feedBack == 2 ) {
                  $resp_data['error'] = 'Error';
                  $_SESSION['stata'] = 'Invoice Sent Successfully ><br>'.$status.' -> '.$result.'<br> but failed to update'.$resp_data['error'];
                  header('Location: ../../syst_tax_soft_transfert_control'); 
                  exit;
              }

              }else{
                  //echo $status.'->'.$object;
                  $_SESSION['stata'] = $status.' -> '.$result;
                  header('Location: ../../syst_tax_soft_transfert_control'); 
                  exit;
              }
     }
*/
?>

