<?php
require_once('../fpdfLib/fpdf.php');
require_once('../fpdfLib/phpqrcode/qrlib.php'); // Ensure phpqrcode is available
class InsurancePDFGeneratorPBMOv2 {
    private $pdf;
    private $data;
    private $headerImage = 'ressources/img/header1s.png';
    private $qrCodeBaseURL = "https://bicnv-sa.com/MGlobal/live/STRv1/ve5/ie1fec5i82eo&1ed96e24y8iy.php";
  

    public function __construct($data) {
        $this->data = $data;
        $this->pdf = new FPDF('P', 'mm', 'A4'); // A4 paper
        $this->pdf->SetMargins(10, 10, 10);
        $this->pdf->AddPage();
        $this->pdf->SetFont('Arial', '', 8);
    }

    public function generate() {
        $this->addHeader();
        $this->addPolicyInfo();
        $this->addGoodsCharacteristics();
        $this->addInsuranceDetails();
        $this->addSignatures();
        return $this->pdf->Output('S');
    }

    /** HEADER: IMAGE + TITLE **/
    private function addHeader() {
        if (file_exists($this->headerImage)) {
            $this->pdf->Image($this->headerImage, 10, 3, 150); // full width
            $this->pdf->Ln(20); // Space after image
        }
        $this->pdf->SetFont('Arial', 'B', 10);
        $this->pdf->Cell(0, 10, 'CONDITIONS PARTICULIERES: ASSURANCE AUTOMOBILE', 0, 1, 'C');
        $this->addHorizontalLine();
    }

    /** POLICY INFO **/
    private function addPolicyInfo() {
        $this->pdf->SetFont('Arial', '', 8);

        $this->addRow('Date de l\'operation:', $this->formatDate($this->data['DateEmission']), 'Intermediaire:', 'N/A');
        $this->addRow('Numero de Police:', $this->data['Numpol'], 'Facture No.:', $this->data['facture']);
        $this->addHorizontalLine();

        $this->addRow('Date d\'effet:', $this->formatDate($this->data['DateEffet']), 'Date d\'echeance:', $this->formatDate($this->data['DateEcheance']));
        $this->addHorizontalLine();

        $this->addRow('Preneur d\'assurance:', $this->data['Proprietaire'], 'Residence:', $this->data['residence']);
        $this->addRow('NIF:', $this->data['NIF'], 'Souscripteur No.:', $this->data['user_code']);
        $this->addRow('Adresse:', $this->data['residence'], 'Telephone:', $this->data['phone']);
    }

    /** VEHICLE INFO **/
    private function addGoodsCharacteristics() {
        $this->addHorizontalLine();
        $this->pdf->SetFont('Arial', 'B', 10);
        $this->pdf->Cell(0, 10, 'CARACTERISTIQUES DES OBJETS ASSURES', 0, 1, 'C');
        $this->addHorizontalLine();
        $this->pdf->SetFont('Arial', '', 8);

        $this->addRow('Immatric. Veh.:', $this->data['Plaque'], 'No Chassis:', $this->data['NumChasis']);
        $this->addRow('Marque:', $this->data['ClassMarkModel'], 'Puissance:', $this->data['PuissanceCV']);
        $this->addRow('Pass. en Cabine:', $this->data['PaleSeatCabine'], 'Pass. sur plateau:', $this->data['SeatPlacePlato']);
        $this->addRow('Usage:', $this->data['UsageAuto'], 'N° Certificat.:', $this->data['NumCertificat']);
    }

    /** INSURANCE DETAILS **/
    private function addInsuranceDetails() {
        $this->addHorizontalLine();
        $this->pdf->SetFont('Arial', 'B', 10);
        $this->pdf->Cell(0, 10, 'DETAILS DE LA PRIME D\'ASSURANCE AUTOMOBILE', 0, 1, 'C');
        $this->addHorizontalLine();
        $this->pdf->SetFont('Arial', 'B', 8);

        $this->pdf->Cell(45, 8, 'Objet assure', 0, 0);
        $this->pdf->Cell(70, 8, 'Garantie', 0, 0);
        $this->pdf->Cell(40, 8, 'Prime Nette', 0, 0);
        $this->pdf->Cell(0, 8, 'Prime Brute', 0, 1);

        $this->pdf->SetFont('Arial', '', 8);
        $this->pdf->Cell(45, 8, $this->data['Plaque'], 0, 0);
        $this->pdf->Cell(70, 8, $this->data['Garantie'], 0, 0);
        $this->pdf->Cell(40, 8, $this->formatNumber($this->data['PN']), 0, 0);
        $this->pdf->Cell(0, 8, $this->formatNumber($this->data['PB']), 0, 1);

        $this->addHorizontalLine();
        
        $this->addRow('PN ET FRAIS HORS TAXE:', $this->formatNumber($this->data['PN']));

        $this->addHorizontalLine();
        $this->addRow('Prime nette:', $this->formatNumber($this->data['PN']), 'Frais admin.:', $this->formatNumber($this->data['FRAIS']));
        $this->addRow('Autres frais (CD):', $this->formatNumber($this->data['CD']), 'TVA:', $this->formatNumber($this->data['TVA']));
        $this->addRow('Rabbais:', '0', 'Modalite de Paiement:', $this->data['mode_payment']);
    }

    /** SIGNATURES **/
    private function addSignatures() {
        $this->addHorizontalLine();
        $this->pdf->SetFont('Arial', 'B', 9);
        $this->pdf->Cell(12, 8, 'N.B.*:', 0, 0);
        $this->pdf->Cell(0, 8, 'La presente police prend effet dans un delai de 6h ou le lendemain a 00h00.', 0, 1);
        $this->addHorizontalLine();
        $this->pdf->SetFont('Arial', '', 9);
        $this->pdf->MultiCell(0, 8, 'Le preneur d\'assurance declare avoir pris connaissance des conditions generales et particulieres qui font partie integrante de la police d\'assurance.', 0, 1);
        $this->pdf->Ln(5);

        $this->pdf->Cell(0, 8, 'Fait a Bujumbura, le ' . $this->formatDate($this->data['DateEmission']), 0, 1, 'C');
        $this->pdf->Ln(3);

        $this->pdf->Cell(95, 8, 'LE PRENEUR', 0, 0, 'C');
        $this->pdf->Cell(95, 8, 'POUR LA COMPAGNIE D\'ASSURANCE', 0, 1, 'C');
        $this->pdf->Ln(5);

        $this->pdf->Cell(95, 8, $this->data['Proprietaire'], 0, 0, 'C');
        $this->pdf->Cell(95, 8, $this->data['Creator'], 0, 1, 'C');
        
        // === Add QR code below on the right side ===
        $qrPath = $this->generateQRCode($this->data['Id'], $this->data['detailsqrCode']);
        $qrX = 150;  // Adjust X to align to the right
        $qrY = $this->pdf->GetY() - 45;  // Position a bit below the text
        $this->pdf->Image($qrPath, $qrX, $qrY, 25, 25); // Size 30x30mm
        
    }

    /** HELPERS **/
    private function addRow($label1, $value1, $label2 = '', $value2 = '') {
        $this->pdf->SetFont('Arial', '', 9);
        $this->pdf->Cell(50, 5, $label1, 0, 0);
        $this->pdf->Cell(50, 5, $value1, 0, 0);
        if ($label2) {
            $this->pdf->Cell(40, 5, $label2, 0, 0);
            $this->pdf->Cell(0, 5, $value2, 0, 1);
        } else {
            $this->pdf->Cell(0, 5, '', 0, 1);
        }
    }
    
    private function generateQRCode($param1, $param2) {
        $fullUrl = $this->qrCodeBaseURL . "?id=" . urlencode($param1) . "&iNvSv=" . urlencode($param2);
        $tempDir = sys_get_temp_dir();
        $filePath = $tempDir . '/qrcode_' . uniqid() . '.png';
        QRcode::png($fullUrl, $filePath, QR_ECLEVEL_H, 4);
        return $filePath;
    }

    private function addHorizontalLine($width = 0.3, $color = [150, 150, 150], $padding = 5) {
        $this->pdf->SetDrawColor($color[0], $color[1], $color[2]);
        $this->pdf->SetLineWidth($width);
        $y = $this->pdf->GetY();
        $this->pdf->Line(10, $y + ($padding / 2), 200, $y + ($padding / 2));
        $this->pdf->Ln($padding);
    }

    private function formatDate($dateString) {
        return !empty($dateString) ? date('d/m/Y', strtotime($dateString)) : '';
    }
    private function formatNumber($number) {
        return number_format($number, 2, ',', ' ');
    }
}
